package org.molap.network

open class SimpleGraph<V,E>(nodes: Collection<Node<V, E>> = emptyList(), edges: Collection<Edge<V, E>> = emptyList(), val directed : Boolean = false) : Graph<V,E> {
    val vertices: MutableMap<V, Node<V, E>> = nodes
        .map { Pair(it.value, it) }
        .toMap(mutableMapOf())
    val edges: MutableList<Edge<V, E>> = edges.toMutableList()

    val edgeToE: MutableMap<E, Edge<V, E>> = edges
        .map { Pair(it.label!!, it) }
        .toMap(mutableMapOf())
    val nodesToE: MutableMap<Link<V, V>, E> = edges
        .map { Pair(DirectedLink(it.n1.value, it.n2.value), it.label!!) }
        .toMap(mutableMapOf())

    override val nodes: Iterable<V>
        get() = vertices.keys

    override val links: Iterable<E> by lazy { this.edges.map { it.label!! } }

    override fun target(link: E, node: V) : V {
        return edgeToE[link]!!.target(vertices[node]!!)!!.value
    }

    override fun getFrom(link: E): V {
        return edgeToE[link]!!.n1.value
    }

    override fun getTo(link: E): V {
        return edgeToE[link]!!.n2.value
    }

    override fun getLink(n1: V, n2: V) : E? {
        if(directed) {
            return nodesToE.get(DirectedLink(n1, n2))
        } else {
            return nodesToE.get(UndirectedLink(n1, n2))
        }
    }

    override fun getLinks(n: V) : Iterable<E> {
        return vertices[n]!!.edges.map { it.label!! }
    }

    override fun getNeighbors(n: V) : Iterable<V> {
        return vertices[n]!!.neighbors().map { it.value }
    }

    override fun toString(): String {
        val standaloneNodes = vertices.values.filter { node -> edges.all { it.n1 != node && it.n2 != node } }
        val s = (edges.map { it.toString() } + standaloneNodes.map { it.toString() }).joinToString()
        return "[$s]"
    }

    protected fun addNode(value: V): Node<V, E> {
        val node = Node<V, E>(value)
        vertices[value] = node
        return node
    }

    protected fun addDirectedEdge(n1: V, n2: V, label: E?) {
        if (!vertices.contains(n1) || !vertices.contains(n2)) {
            throw IllegalStateException("Expected '$n1' and '$n2' nodes to exist in graph")
        }
        val edge = DirectedEdge(vertices[n1]!!, vertices[n2]!!, label)
//        if (edges.none { it.equivalentTo(edge) }) {
        edges.add(edge)
        vertices[n1]!!.edges.add(edge)
        vertices[n2]!!.edges.add(edge)
//        } else {
//            throw IllegalStateException("Expected edge between '$n1' and '$n2' not to already exist in graph")
//        }

        if(label != null) {
            edgeToE.put(label, edge)
            nodesToE.put(DirectedLink(n1, n2), label)
        }
    }

    protected fun addUndirectedEdge(n1: V, n2: V, label: E?) {
        if (!vertices.contains(n1) || !vertices.contains(n2)) {
            throw IllegalStateException("Expected '$n1' and '$n2' nodes to exist in graph")
        }
        val edge = UndirectedEdge(vertices[n1]!!, vertices[n2]!!, label)
//        if (edges.none { it.equivalentTo(edge) }) {
            edges.add(edge)
            vertices[n1]!!.edges.add(edge)
            vertices[n2]!!.edges.add(edge)
//        } else {
//            throw IllegalStateException("Expected edge between '$n1' and '$n2' not to already exist in graph")
//        }

        if(label != null) {
            edgeToE.put(label, edge)
            nodesToE.put(UndirectedLink(n1, n2), label)
        }
    }

    data class Node<V, L>(val value: V) {
        val edges: MutableList<Edge<V, L>> = ArrayList()
        fun neighbors(): List<Node<V, L>> = edges.map { edge -> edge.target(this)!! }
        override fun toString() = value.toString()
    }

    interface Edge<V, L> {
        val n1: Node<V, L>
        val n2: Node<V, L>
        val label: L?
        fun target(node: Node<V, L>): Node<V, L>?
        fun equivalentTo(other: Edge<V, L>) =
            (n1 == other.n1 && n2 == other.n2) || (n1 == other.n2 && n2 == other.n1)
    }

    data class UndirectedEdge<V, L>(override val n1: Node<V, L>, override val n2: Node<V, L>, override val label: L?): Edge<V, L> {
        override fun target(node: Node<V, L>) = if (n1 == node) n2 else if (n2 == node) n1 else null
        override fun toString() = "$n1-$n2${if (label == null) "" else "/$label"}"
    }

    data class DirectedEdge<V, L>(override val n1: Node<V, L>, override val n2: Node<V, L>, override val label: L?): Edge<V, L> {
        override fun target(node: Node<V, L>) = if (n1 == node) n2 else null
        override fun toString() = "$n1>$n2${if (label == null) "" else "/$label"}"
    }

    companion object {
        fun <V, L> labeledTerms(termForm: TermForm<V, L>): SimpleGraph<V, L> =
            createFromTerms(termForm) { graph, n1, n2, value -> graph.addUndirectedEdge(n1, n2, value) }


        fun <V> adjacent(adjacencyList: AdjacencyList<V, Nothing>): SimpleGraph<V, *> =
            fromAdjacencyList(adjacencyList) { graph, n1, n2, value ->
                graph.addUndirectedEdge(n1, n2, value)
            }

        fun <V, L> labeledAdjacent(adjacencyList: AdjacencyList<V, L>): SimpleGraph<V, L> =
            fromAdjacencyList(adjacencyList) { graph, n1, n2, value ->
                graph.addUndirectedEdge(n1, n2, value)
            }

        private fun <V, L> createFromTerms(termForm: TermForm<V, L>, addFunction: (SimpleGraph<V, L>, V, V, L?) -> Unit): SimpleGraph<V, L> {
            val graph = SimpleGraph<V, L>()
            termForm.nodes.forEach { graph.addNode(it) }
            termForm.edges.forEach { addFunction(graph, it.n1, it.n2, it.label) }
            return graph
        }

        private fun <V, L> fromAdjacencyList(adjacencyList: AdjacencyList<V, L>, addFunction: (SimpleGraph<V, L>, V, V, L?) -> Unit): SimpleGraph<V, L> {
            val graph = SimpleGraph<V, L>()
            adjacencyList.entries.forEach { graph.addNode(it.node) }
            adjacencyList.entries.forEach { (node, links) ->
                links.forEach { addFunction(graph, node, it.node, it.label) }
            }
            return graph
        }
    }
}