/*
 * Copyright (c) 2017 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.subset

import com.macrofocus.common.collection.*
import com.macrofocus.common.filter.Filter
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.selection.SelectionEvent
import com.macrofocus.common.selection.SelectionListener
import com.macrofocus.common.selection.SimpleSelection
import kotlin.jvm.Synchronized
import kotlin.math.max

/**
 * Created by luc on 07.04.17.
 */
abstract class AbstractBinningDimension<Row, Column, Value, Bin>(
    dataFrame: SubsetDataFrame<Row, Column, Value>,
    filteringCallback: FilteringCallback<Row>,
    selection: MutableSelection<Row>?,
    operation: Operation
) : AbstractDimension<Row>(), BinningDimension<Row, Bin> {
    enum class Operation {
        And {
            override fun mixSortedArrays(union: IntArray?, rows: IntArray?): IntArray {
                return SortedArraySetOperations.intersectSortedArrays(union, rows)
            }
        },
        Or {
            override fun mixSortedArrays(union: IntArray?, rows: IntArray?): IntArray {
                return SortedArraySetOperations.unionSortedArrays(union, rows)
            }
        };

        abstract fun mixSortedArrays(union: IntArray?, rows: IntArray?): IntArray
    }

    private val operation: Operation
    protected var dataFrame: SubsetDataFrame<Row, Column, Value>
    protected var index: Map<Bin, IntArray>? = null
    get() {
        return if (field != null) {
            field
        } else {
            materializeIndex()
            field
        }
    }
    protected var maxValueCount = 0
    protected var sumValueCount = 0
    protected var filterCounts: ObjectIntMap<Bin>?
    protected var selectedCounts: ObjectIntMap<Bin>
    private var maxActiveCount = -1.0
    private var sumActiveCount = -1.0
    override var bins: RandomAccessIterable<Bin>? = null
        get() {
            return if (activeBins != null) activeBins else {
                if (field == null) {
                    materializeIndex()
                }
                field!!
            }
        }
    override var activeBins: RandomAccessIterable<Bin>? = null
    private var dirty = false
    protected lateinit var otherActiveIndices: IntArray
    override var activeIndices: IntArray? = null
        get() {
            if (dirty) {
                field = updateActiveIndices()
                dirty = false
            }
            return field
        }
    protected var groups: MutableMap<Reducer<Row, Bin>, Group<Row, Bin>>? = null
    private val activeSelection: MutableSelection<Bin> = SimpleSelection()
    override var inverseFilter = false
        get() {
            return field
        }
        set(value) {
            if (field != value) {
                field = value
                scheduleUpdateFilter()
            }
        }
    private val filteringCallback: FilteringCallback<Row>
    override val selection: MutableSelection<Row>?
    protected var updateFilterImmediately = false
    override val isReducable: Boolean
        get() = true

    protected fun updateActive() {
        var maxActiveCount = 0.0
        var sumActiveCount = 0.0
        val activeBins: RandomAccessIterable<Bin>
        if (filterCounts != null && filterCounts!!.size() > 0) {
            val list: ArrayList<Bin> = ArrayList<Bin>()
            activeBins = list.asRandomAccessIterable()
            for (key in bins!!) {
                val activeCount = getActiveDensity(key)
                maxActiveCount = max(activeCount, maxActiveCount)
                sumActiveCount += activeCount
                if (activeCount > 0) {
                    list.add(key)
                }
            }
        } else {
            maxActiveCount = maxDensity
            sumActiveCount = sumDensity
            activeBins = bins!!
        }

//        if(activeBins.size() < 1) {
//            System.out.println(column + " has not active value " + maxActiveCount + ", " + filterCounts.size());
//        }
        this.activeBins = activeBins
        this.maxActiveCount = maxActiveCount
        this.sumActiveCount = sumActiveCount
    }

    override fun getRows(bin: Bin): Iterable<Row> {
        return Iterables.conditional(IndicesIterable<Row>(dataFrame.unfilteredDataFrame, object : IndicesSupplier {
            override fun get(): IntArray? {
                return index!![bin]
            }
        }), object : Iterables.Condition<Row> {
            override fun apply(row: Row): Boolean {
                return !filter!!.isFiltered(row)
            }
        })
    }

    val filter: Filter<Row>?
        get() = dataFrame.getOutputFilter()
    override val filterExact: MutableSelection<Bin>
        get() = activeSelection

    override fun filterExact(vararg bins: Bin) {
//            long now = System.currentTimeMillis();
        filterExact.setSelectedElements(*bins)

//            System.out.println("filterExact of " + Arrays.toString(values) + " executed in " + (System.currentTimeMillis() - now) + " ms");
    }

    //    @Override
    override fun filterExactIterable(bins: Iterable<Bin>) {
        filterExact.setSelectedIterable(bins)
    }

    override fun filterAll() {
//            long now = System.currentTimeMillis();
        filterExact.clearSelection()

//            System.out.println("filterAll executed in " + (System.currentTimeMillis() - now) + " ms");
    }

    @Synchronized
    protected override fun scheduleUpdateFilter() {
        if (updateFilterImmediately || updateTimer == null) {
            updateFilter()
        } else {
            updateTimer!!.restart()
        }
    }

    @Synchronized
    protected override fun updateFilter() {
        val oldIndices = activeIndices
        activeIndices = updateActiveIndices()
        val newIndices = activeIndices

//        if(filtered != null || unfiltered != null) {
        filteringCallback.filteringChanged(FilteringEvent<Row>(this, IndicesIterable<Row>(dataFrame.unfilteredDataFrame, object : IndicesSupplier {
            override fun get(): IntArray? {
                return dataFrame.computeFiltered(oldIndices, newIndices)
            }
        }), IndicesIterable<Row>(dataFrame.unfilteredDataFrame, object : IndicesSupplier {
            override fun get(): IntArray? {
                return dataFrame.computeUnfiltered(oldIndices, newIndices)
            }
        })))
        //        }
    }

    override fun getDensity(bin: Bin): Double {
//        assert(getIndex()!!.containsKey(bin)) { bin.toString() + " not found for " + getName() }
        return index!![bin]!!.size.toDouble()
    }

    override fun getActiveDensity(bin: Bin): Double {
        val activeCount = getDensity(bin) - getFilterDensity(bin)
//        assert(activeCount >= 0)
//        assert(activeCount <= getDensity(bin))
        return activeCount
    }

    override fun getFilterDensity(bin: Bin): Double {
        return filterCounts!!.get(bin).toDouble()
    }

    override fun getSelectionDensity(bin: Bin): Double {
        return selectedCounts.get(bin).toDouble()
    }

    override val maxDensity: Double
        get() {
            index
            return maxValueCount.toDouble()
        }

    // If maxActiveCount has not been initialized, maxValueCount is returned
    override val maxActiveDensity: Double
        get() {
            index

            // If maxActiveCount has not been initialized, maxValueCount is returned
            return if (maxActiveCount >= 0) maxActiveCount else maxDensity
        }
    override val sumDensity: Double
        get() {
            index
            return sumValueCount.toDouble()
        }

    // If sumActiveCount has not been initialized, sumValueCount is returned
    override val sumActiveDensity: Double
        get() {
            index

            // If sumActiveCount has not been initialized, sumValueCount is returned
            return if (sumActiveCount >= 0) sumActiveCount else sumDensity
        }

    override fun markDirty() {
        index = null
        bins = null
        activeBins = null
        maxActiveCount = 0.0
        maxValueCount = 0
        sumActiveCount = 0.0
        sumValueCount = 0
        dirty = true
    }

    open fun updateActiveIndices(): IntArray? {
        return if (!inverseFilter) {
            if (filterExact.isActive) {
                val index = index
//                assert(index != null) { "Dimension \"$this\" is not indexed" }
                var union: IntArray? = null
                for (value in filterExact) {
                    val rows = index!![value]

                    // If the value defined by the criteria can be found in the data
                    if (rows != null) {
                        union = if (union == null) {
                            rows
                        } else {
                            operation.mixSortedArrays(union, rows)
                        }
                    } else {
                        // Filtered value could not be found in bins, filter everything!
                        union = IntArray(0)
                        break
                    }
                }
                union
            } else {
                null
            }
        } else {
            if (filterExact.isActive) {
                val index = index
//                assert(index != null) { "Dimension \"$this\" is not indexed" }
                var union: IntArray? = null
                for (value in filterExact) {
                    val rows = index!![value]

                    // If the value defined by the criteria can be found in the data
                    if (rows != null) {
                        val diff: IntArray = SortedArraySetOperations.diffSortedTIntArrays(dataFrame.unfilteredDataFrame.rowCount, rows)
                        union = if (union == null) {
                            diff
                        } else {
                            operation.mixSortedArrays(union, diff)
                        }
                    } else {
                        // Filtered value could not be found in bins, filter everything!
                        union = null
                        break
                    }
                }
                union
            } else {
                IntArray(0)
            }
        }
    }

    // If the value defined by the criteria can be found in the data
    override val activeIndicesUsingArrays: IntArray?
        get() = if (filterExact.isActive) {
            val index = index
//            assert(index != null) { "Dimension \"$this\" is not indexed" }
            var union: IntArray? = null
            for (value in filterExact) {
                val rows = index!![value]

                // If the value defined by the criteria can be found in the data
                if (rows != null) {
                    union = if (union == null) {
                        rows
                    } else {
                        operation.mixSortedArrays(union, rows)
                    }
                }
            }
            union
        } else {
            null
        }

    // If the value defined by the criteria can be found in the data
    override val activeIndicesUsingHashSet: TIntSet?
        get() = if (filterExact.isActive) {
            val index = index
//            assert(index != null) { "Dimension \"$this\" is not indexed" }
            val union: TIntSet = TIntHashSet()
            for (value in filterExact) {
                val rows = index!![value]

                // If the value defined by the criteria can be found in the data
                if (rows != null) {
                    union.addAll(rows.toTypedArray())
                }
            }
            union
        } else {
            null
        }

    init {
        this.dataFrame = dataFrame
        this.filteringCallback = filteringCallback
        this.selection = selection
        this.operation = operation
        val bins: RandomAccessIterable<Bin>? = bins
        filterCounts = ObjectIntMap(if (bins != null) bins.size() else 10, 0)
        selectedCounts = ObjectIntMap(if (bins != null) bins.size() else 10, 0)
        activeSelection.addSelectionListener(object : SelectionListener<Bin> {
            override fun selectionChanged(event: SelectionEvent<Bin>) {
                scheduleUpdateFilter()
            }
        })
    }
}