/*
 * Copyright (c) 2015 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.subset

import com.macrofocus.common.collection.CollectionFactory.copyOnWriteArrayList
import com.macrofocus.common.collection.WeakReference
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import org.molap.dataframe.DataFrame

/**
 * Created by luc on 16/05/15.
 */
abstract class AbstractDimension<Row> : Dimension<Row> {
    override var updateTimer: CPTimer? = null
    set(value) {
        field?.removeTimerListener(updateTimerListener)
        field = value
        field?.addTimerListener(updateTimerListener)
    }
    private val listeners: MutableList<DimensionListener<Row>>
    private val updateTimerListener: CPTimerListener = object : CPTimerListener {
        override fun timerTriggered() {
            updateFilter()
        }
    }

    protected open fun scheduleUpdateFilter() {
        if (updateTimer != null) {
            updateTimer!!.restart()
        } else {
            updateFilter()
        }
    }

    protected abstract fun updateFilter()

    override fun addDimensionListener(listener: DimensionListener<Row>) {
        listeners.add(listener)
    }

    override fun addWeakDimensionListener(listener: DimensionListener<Row>) {
        val weakListener: WeakDimensionListener = WeakDimensionListener(listener)
        listeners.add(weakListener)
    }

    override fun removeDimensionListener(listener: DimensionListener<Row>) {
        if (listener is WeakDimensionListener) {
            val removed = listeners.remove(listener)
//                assert(removed) { listener }
        } else {
            var toRemove: DimensionListener<Row>? = null
            for (dimensionListener in listeners) {
                var comparable: DimensionListener<*>?
                comparable = if (dimensionListener is WeakDimensionListener) {
                    dimensionListener.reference
                } else {
                    dimensionListener
                }
                if (listener == comparable) {
                    toRemove = dimensionListener
                }
            }
            if (toRemove != null) {
                val removed = listeners.remove(toRemove)
//                    assert(removed) { listener }
            }
        }
    }

    override fun removeDimensionListeners() {
        listeners.clear()
    }

    protected fun notifyDimensionChanged(event: DimensionEvent<Row>) {
        for (listener in listeners) {
            listener.dimensionChanged(event)
        }
    }

    protected fun notifySelectedCountChanged() {
        for (listener in listeners) {
            listener.selectedCountChanged()
        }
    }

    interface FilteringCallback<R> {
        fun filteringChanged(event: FilteringEvent<R>)
    }

    class FilteringEvent<Row>(dimension: Dimension<Row>, filtered: Iterable<Row>, unfiltered: Iterable<Row>) {
        private val dimension: Dimension<Row>
        val filtered: Iterable<Row>
        val unfiltered: Iterable<Row>
        fun getDimension(): Dimension<Row> {
            return dimension
        }

        init {
            this.dimension = dimension
            this.filtered = filtered
            this.unfiltered = unfiltered
        }
    }

    class IndicesIterable<R>(private val dataFrame: DataFrame<R, *, *>, private val indices: IndicesSupplier) : Iterable<R> {
        constructor(dataFrame: DataFrame<R, *, *>, indices: IntArray?) : this(dataFrame, object : IndicesSupplier {
            override fun get(): IntArray? {
                return indices
            }
        }) {
        }

        override fun iterator(): Iterator<R> {
            return object : MutableIterator<R> {
                var i = 0
                override fun hasNext(): Boolean {
                    return indices.get() != null && i < indices.get()!!.size
                }

                override fun next(): R {
                    val key = dataFrame.getRowKey(indices.get()!![i])
                    i++
                    return key
                }

                override fun remove() {
                    throw UnsupportedOperationException()
                }
            }
        }

        override fun toString(): String {
            return "IndicesIterable{" +
                    "indices=" + (if (indices.get() != null) indices.get()!!.size else "(null)") +
                    '}'
        }
    }

    interface IndicesSupplier {
        fun get(): IntArray?
    }

    private inner class WeakDimensionListener(listener: DimensionListener<Row>) : DimensionListener<Row> {
        private val l_ref: WeakReference<DimensionListener<Row>>
        override fun dimensionChanged(event: DimensionEvent<Row>) {
            val l: DimensionListener<Row>? = reference
            if (l != null) {
                l.dimensionChanged(event)
            } else {
                removeDimensionListener(this as DimensionListener<Row>)
            }
        }

        override fun selectedCountChanged() {
            val l: DimensionListener<Row>? = reference
            if (l != null) {
                l.selectedCountChanged()
            } else {
                removeDimensionListener(this as DimensionListener<Row>)
            }
        }

        internal val reference: DimensionListener<Row>?
            get() = l_ref.get()

        override fun toString(): String {
            val l: DimensionListener<Row>? = reference
            return if (l != null) {
                "Weak[$l]"
            } else {
                super.toString()
            }
        }

        init {
            l_ref = WeakReference<DimensionListener<Row>>(listener)
        }
    }

    init {
        listeners = copyOnWriteArrayList<DimensionListener<Row>>()
    }
}