/*
 * Copyright (c) 2017 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.subset

import com.macrofocus.common.selection.*
import org.molap.subset.MultiBinningDimension.SortedArrayFilter

/**
 * Created by luc on 10.04.17.
 */
abstract class AbstractMultiBinningDimension<Row, Column, Value, Bin>(
    dataFrame: SubsetDataFrame<Row, Column, Value>,
    filteringCallback: FilteringCallback<Row>,
    selection: MutableSelection<Row>?,
    operation: Operation
) : AbstractBinningDimension<Row, Column, Value, Bin>(dataFrame, filteringCallback, selection, operation),
    MultiBinningDimension<Row, Bin> {
    private val filterSelection: MutableSingleSelection<SortedArrayFilter<Bin>>? = SimpleSingleSelection<SortedArrayFilter<Bin>>()
    override val defaultSortedArrayFilter: DefaultSortedArrayFilter = DefaultSortedArrayFilter()
    override fun updateActiveIndices(): IntArray? {
        return if (filterSelection != null && filterSelection.isActive) {
            filterSelection.selected!!.evaluate()
        } else {
            super.updateActiveIndices()
        }
    }

    override fun createSortedArrayFilter(bin: Bin): SortedArrayFilter<Bin> {
        return BinSortedArrayFilter(bin)
    }

    fun getFilterSelection(): MutableSingleSelection<SortedArrayFilter<Bin>>? {
        return filterSelection
    }

    abstract inner class AbstractSortedArrayFilter : SortedArrayFilter<Bin> {
        override fun or(others: Iterable<SortedArrayFilter<Bin>>): SortedArrayFilter<Bin> {
            return OrSortedArrayFilter(this, others)
        }

        override fun or(other: SortedArrayFilter<Bin>): SortedArrayFilter<Bin> {
            return OrSortedArrayFilter(this, other)
        }

        override fun and(other: SortedArrayFilter<Bin>): SortedArrayFilter<Bin> {
            return AndSortedArrayFilter(this, other)
        }

        override fun and(others: Iterable<SortedArrayFilter<Bin>>): SortedArrayFilter<Bin> {
            return OrSortedArrayFilter(this, others)
        }
    }

    inner class DefaultSortedArrayFilter : AbstractSortedArrayFilter() {
        override fun evaluate(): IntArray? {
            return super@AbstractMultiBinningDimension.updateActiveIndices()
        }

        override fun toString(): String {
            return "Default"
        }
    }

    inner class BinSortedArrayFilter(private val bin: Bin) : AbstractSortedArrayFilter() {
        override fun evaluate(): IntArray? {
            return if (!inverseFilter) {
                val rows: IntArray? = index!!.get(bin)
                rows ?: IntArray(0)
            } else {
                val rows: IntArray? = index!!.get(bin)
                if (rows != null) {
                    SortedArraySetOperations.diffSortedTIntArrays(dataFrame.unfilteredDataFrame.rowCount, rows)
                } else {
                    null
                }
            }
        }

        override fun toString(): String {
            return bin.toString()
        }
    }

    inner class OrSortedArrayFilter : AbstractSortedArrayFilter {
        private var first: SortedArrayFilter<Bin>?
        private var filters: Iterable<SortedArrayFilter<Bin>>

        constructor(first: SortedArrayFilter<Bin>, filters: Iterable<SortedArrayFilter<Bin>>) {
            this.first = first
            this.filters = filters
        }

        constructor(first: SortedArrayFilter<Bin>, vararg filters: SortedArrayFilter<Bin>) {
            this.first = first
            this.filters = filters.asIterable()
        }

        override fun evaluate(): IntArray? {
//            assert(first != null)
            return if (!inverseFilter) {
                var current: IntArray? = first!!.evaluate()
                if (current != null) {
                    for (filter in filters) {
                        val evaluation: IntArray = filter.evaluate() ?: return null
                        current = SortedArraySetOperations.unionSortedArrays(current, evaluation)
                    }
                    current
                } else {
                    null
                }
            } else {
                throw UnsupportedOperationException("Not yet implemented")
            }
        }

        override fun toString(): String {
            val sb: StringBuilder = StringBuilder("Or(")
            sb.append(first)
            for (filter in filters) {
                sb.append(" ").append(filter)
            }
            sb.append(')')
            return sb.toString()
        }
    }

    inner class AndSortedArrayFilter : AbstractSortedArrayFilter {
        private var first: SortedArrayFilter<Bin>?
        private var filters: Iterable<SortedArrayFilter<Bin>>

        constructor(first: SortedArrayFilter<Bin>, filters: Iterable<SortedArrayFilter<Bin>>) {
            this.first = first
            this.filters = filters
        }

        constructor(first: SortedArrayFilter<Bin>, vararg filters: SortedArrayFilter<Bin>) {
            this.first = first
            this.filters = filters.asIterable()
        }

        override fun evaluate(): IntArray? {
//            assert(first != null)
            return if (!inverseFilter) {
                var current: IntArray? = first!!.evaluate()
                for (filter in filters) {
                    val evaluation: IntArray? = filter.evaluate()
                    if (current != null) {
                        if (evaluation != null) {
                            current = SortedArraySetOperations.intersectSortedArrays(current, evaluation)
                        }
                    } else {
                        current = evaluation
                    }
                }
                current
            } else {
                throw UnsupportedOperationException("Not yet implemented")
            }
        }

        override fun toString(): String {
            val sb: StringBuilder = StringBuilder("And(")
            sb.append(first)
            for (filter in filters) {
                sb.append(" ").append(filter)
            }
            sb.append(')')
            return sb.toString()
        }
    }

    init {
        filterSelection!!.addSingleSelectionListener(object : SingleSelectionListener<SortedArrayFilter<Bin>> {
            override fun selectionChanged(event: SingleSelectionEvent<SortedArrayFilter<Bin>>) {
                scheduleUpdateFilter()
            }
        })
    }
}