package org.molap.exporter

import com.macrofocus.common.units.Quantity
import org.molap.dataframe.DataFrame
import java.io.IOException
import java.util.*
import kotlin.reflect.KClass


class JSONDataFrameWriter(private val output: DataFrameOutput, newLine: String) : AbstractDataFrameWriter() {
    private val columnNames: MutableMap<Int, String>
    private val newLine: String
    private var row = 0
    private var column = 0
    private var first = true
    @JvmOverloads
    @Throws(IOException::class)
    override fun start() {
        start("data")
    }

    @JvmOverloads
    @Throws(IOException::class)
    fun start(name: String) {
        if (!first) {
            output.write(",")
            output.write(newLine)
        } else {
            first = false
        }
        output.write("\t\"$name\": [")
    }


    @Throws(IOException::class)
    override fun writeColumnName(name: String?, hasMore: Boolean) {
        columnNames[column] = name!!
        column++
    }

    override fun includeType(): Boolean {
        return false
    }

    @Throws(IOException::class)
    override fun writeType(name: KClass<*>, hasMore: Boolean) {
    }

    @Throws(IOException::class)
    override fun <Row,Column,Value> writeCell(value: Any?, dataFrame: DataFrame<Row,Column,Value>, rowKey: Row, columnKey: Column) {
        if (column == 0) {
            if (row > 1) {
                output.write("},")
            }
            output.write(newLine)
            output.write("\t\t{")
        } else {
            output.write(", ")
        }
        var str: String
        if (value == null) {
            str = "null"
        } else if (value is Date) {
            str = "\"" + dateToString(dataFrame, columnKey, value as Date?) + "\""
        } else if (value is Number) {
            str = numberToString(value as Number?)
        } else if(value is Quantity<*>) {
            str = numberToString(value.amount)
        } else if (value is Boolean) {
            str = value.toString()
        } else {
            if (value is Array<*>) {
                val array = value
                str = "["
                for (i in array.indices) {
                    val s = array[i]
                    str += "\"" + quote(s as String) + "\""
                    if (i < array.size - 1) {
                        str += ", "
                    }
                }
                str += "]"
            } else {
                str = value.toString()
                str = if (str.length > 0) {
                    "\"" + quote(str) + "\""
                } else {
                    "null"
                }
            }
        }
        output.write("\"" + columnNames[column] + "\": ")
        output.write(str)
    }

    @Throws(IOException::class)
    override fun nextColumn(hasMore: Boolean) {
        column++
    }

    @Throws(IOException::class)
    override fun nextRow() {
        column = 0
        row++
    }

    @Throws(IOException::class)
    fun end() {
        if (row > 2) {
            output.write("}")
        }
        output.write(newLine)
        output.write("\t]")
        row = 0
        column = 0
    }

    @Throws(IOException::class)
    override fun close() {
        end()
        terminate()
    }

    @Throws(IOException::class)
    fun terminate() {
        output.write(newLine)
        output.write("}")
        output.close()
    }

    companion object {
        fun quote(string: String?): String {
            if (string == null || string.length == 0) {
                return ""
            }
            var c = 0.toChar()
            var i: Int
            val len = string.length
            val sb = StringBuilder(len + 4)
            var t: String
            i = 0
            while (i < len) {
                c = string[i]
                when (c) {
                    '\\', '"' -> {
                        sb.append('\\')
                        sb.append(c)
                    }
                    '/' -> {
                        //                if (b == '<') {
                        sb.append('\\')
                        //                }
                        sb.append(c)
                    }
                    '\b' -> sb.append("\\b")
                    '\t' -> sb.append("\\t")
                    '\n' -> sb.append("\\n")
//                    '\f' -> sb.append("\\f")
                    '\r' -> sb.append("\\r")
                    else -> if (c < ' ') {
                        t = "000" + Integer.toHexString(c.toInt())
                        sb.append("\\u" + t.substring(t.length - 4))
                    } else {
                        sb.append(c)
                    }
                }
                i += 1
            }
            return sb.toString()
        }
    }

    init {
        columnNames = HashMap()
        this.newLine = newLine
        output.write("{")
        output.write(newLine)
    }
}
