/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates.query

import com.macrofocus.common.collection.sort
import org.molap.aggregates.aggregation.AbstractAggregation
import org.molap.aggregates.aggregation.Aggregation
import org.molap.aggregates.cube.Dimensions
import org.molap.aggregates.cube.Group
import org.molap.aggregates.cuboid.Cuboid
import kotlin.reflect.KClass

class PivotQuery : AbstractQuery {
    private val query: Query
    override var cuboid: Cuboid? = null
        get() {
            if (isDirty) {
                run()
            }
            return field
        }
    private val aggregation: Aggregation<Any?>
    override var aggregations: Array<Aggregation<Any?>>
    get() {
        if (isDirty) {
            run()
        }
        return field
    }
    private var columns: Array<Any?>?
    override var groups: List<Group>? = null
        get() {
            if (isDirty) {
                run()
            }
            return field
        }
    val queryListener: QueryListener = object : QueryListener {
        override fun queryChanged() {
            isDirty = true
            notifyQueryChanged()
        }
    }

    constructor(query: Query, aggregation: Aggregation<Any?>) {
        this.query = query
        this.aggregation = aggregation
        columns = null
        query.addWeakQueryListener(queryListener)
        aggregations = emptyArray()
    }

    constructor(query: Query, aggregation: Aggregation<Any?>, columns: Array<Any?>?) {
        this.query = query
        this.aggregation = aggregation
        this.columns = columns
        query.addWeakQueryListener(queryListener)
        aggregations = emptyArray()
    }

    private fun run() {
        isDirty = false
        val cuboid: Cuboid? = query.cuboid!!.drillUp()
        val groups: MutableList<Group> = ArrayList<Group>()
        for (group in cuboid!!.groups!!) {
            groups.add(group)
        }
        val values: Array<Any?>
        if (columns != null) {
            values = columns!!
        } else {
            val v: Set<Any?>? = query.cuboid!!.values
            values = v!!.toTypedArray()
            sort<Any?>(values, object : Comparator<Any?> {
                override fun compare(o1: Any?, o2: Any?): Int {
                    return if (o1 == null) {
                        if (o2 == null) {
                            0
                        } else {
                            1
                        }
                    } else {
                        if (o2 == null) {
                            -1
                        } else {
                            (o1 as Comparable<Any?>).compareTo(o2)
                        }
                    }
                }
            })
        }
        val aggregations: Array<Aggregation<Any?>?> = arrayOfNulls<Aggregation<Any?>>(values.size)
        for (i in values.indices) {
            val o = values[i]
            aggregations[i] = PivotAggregation(o, aggregation)
        }
        this.cuboid = cuboid
        this.groups = groups
        this.aggregations = aggregations.requireNoNulls()
    }

    override fun getValue(path: Group?, aggregation: Aggregation<Any?>): Any? {
        return aggregation.compute(query.cuboid, path)
    }

    override val dimensions: Dimensions?
        get() = query.dimensions!!.drillUp()

    override fun setDice(valuesSets: Set<Any?>) {
        cuboid = cuboid!!.dice(valuesSets)
        isDirty = true
        notifyQueryChanged()
    }

    private class PivotAggregation(private val value: Any?, aggregation: Aggregation<out Any?>) : AbstractAggregation<Any?>() {
        private val aggregation: Aggregation<out Any?>
        override val type: KClass<Any>
            get() = aggregation.type

        override fun compute(cuboid: Cuboid?, group: Group?): Any? {
            return if (group != null) {
                val g: Group = group.drillDown(value)
                aggregation.compute(cuboid, g)
            } else {
                null
            }
        }

        override fun toString(): String {
            return value.toString()
        }

        init {
            this.aggregation = aggregation
        }
    }
}