/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates.query

import org.molap.aggregates.AggregateDataFrame
import org.molap.aggregates.aggregation.Aggregation
import org.molap.aggregates.cube.Group
import org.molap.aggregates.cuboid.Cuboid
import org.molap.dataframe.*
import org.molap.index.DefaultUniqueIndex
import org.molap.index.MultiKey
import org.molap.index.UniqueIndex
import org.molap.series.AbstractSeries
import org.molap.series.Series
import kotlin.reflect.KClass

class QueryDataFrame<C>(query: Query, includeIndex: Boolean) : AbstractDataFrame<Group?, String?, Any?>(), AggregateDataFrame<C> {
    private val query: Query
    private val includeIndex: Boolean
    override val rowIndex: UniqueIndex<Group?>
        get() {
            return cacheRowIndex!!
        }
    override val columnIndex: UniqueIndex<String?>
        get() {
            return cacheColumnIndex!!
        }
    private var cacheRowIndex: UniqueIndex<Group?>? = null
    get() {
        if (field == null) {
            val groups: Array<Group?> = arrayOfNulls<Group>(query.groups!!.size)
            var i = 0
            for (g in query.groups!!) {
                groups[i++] = g as Group
            }
            field = DefaultUniqueIndex<Group?>(*groups)
        }
        return field
    }
    private var cacheColumnIndex: UniqueIndex<String?>? = null
    get() {
        if (field == null) {
            val names = arrayOfNulls<String>(columnCount)
            for (c in 0 until columnCount) {
                names[c] = getColumnKey(c)
            }
            field = DefaultUniqueIndex<String?>(*names)
        }
        return field
    }
    private val queryListener: QueryListener = object : QueryListener {
        override fun queryChanged() {
            cacheRowIndex = null
            cacheColumnIndex = null
            notifyDataFrameChanged(DataFrameEvent<Group?, String?>(emptyList(), null, true))
        }
    }
    override val cuboid: Cuboid?
        get() = query.cuboid

    override fun drillDown(vararg columns: C): AggregateDataFrame<C> {
        return QueryDataFrame<C>(query.drillDown(*columns), includeIndex)
    }

    override fun drillUp(): AggregateDataFrame<C> {
        return QueryDataFrame<C>(query.drillUp(), includeIndex)
    }

    override fun slice(value: Any?): AggregateDataFrame<C> {
        return QueryDataFrame<C>(query.slice(value), includeIndex)
    }

    override fun dice(valuesSets: Set<Any?>?): AggregateDataFrame<C> {
        return QueryDataFrame<C>(query.dice(valuesSets), includeIndex)
    }

    override fun collapse(): AggregateDataFrame<C> {
        return QueryDataFrame<C>(query.collapse(), includeIndex)
    }

    override fun order(vararg aggregations: Aggregation<Any?>?): AggregateDataFrame<C> {
        return QueryDataFrame<C>(query.order(*aggregations), includeIndex)
    }

    override fun pivot(aggregation: Aggregation<Any?>?): AggregateDataFrame<C> {
        return QueryDataFrame<C>(query.pivot(aggregation), includeIndex)
    }

    override fun on(vararg cuboids: Cuboid): AggregateDataFrame<C> {
        return QueryDataFrame<C>(query.on(*cuboids), includeIndex)
    }

    override fun getRowClass(row: Group?): KClass<*>? {
        return Any::class
    }

    override fun getColumnClass(column: String?): KClass<out Any> {
        var columnIndex: Int = columnIndex.getAddress(column!!)
        return if (columnIndex >= 0) {
            if (includeIndex) {
                if (columnIndex < query.dimensions!!.path.size) {
                    val c = query.dimensions!!.path.get(columnIndex)
                    val dataFrame = query.cuboid!!.cube!!.dataFrame
                    return dataFrame.getColumnClass(c)
                }
                columnIndex -= query.dimensions!!.path.size
            }
            query.aggregations.get(columnIndex).type
        } else {
            Any::class
        }
    }

    override fun getValueAt(path: Group?, column: String?): Any? {
        var columnIndex: Int? = columnIndex.getAddress(column!!)
        return if (columnIndex == null || columnIndex >= 0) {
            if (path == null) {
                println("QueryDataFrame: path is null")
            }
//            assert(path != null)
            if (includeIndex) {
                if (columnIndex!! < query.dimensions!!.path.size) {
                    if (query.dimensions != null) {
                        val c: Any? = query.dimensions!!.path.get(columnIndex)
                        columnIndex = query.dimensions!!.getIndex(c)
                    }
                    return if (path != null) {
                        path.getPath(columnIndex!!)
                    } else {
                        null
                    }
                }
                columnIndex -= query.dimensions!!.path.size
            }
            query.getValue(path, query.aggregations.get(columnIndex!!))
        } else {
            null
        }
    }

    fun getRow(row: Group?): Series<String, *>? {
        return null
    }

    override fun getColumn(column: String?): Series<Group?, Any?> {
        var columnIndex: Int = columnIndex.getAddress(column!!)
        if (includeIndex) {
            if (columnIndex < query.dimensions!!.path.size) {
                return DimensionSeries(query.cuboid!!.cube!!.dataFrame as DataFrame<Group, String, Any>, column, columnIndex)
            }
            columnIndex -= query.dimensions!!.path.size
        }
        return AggregationSeries(query.aggregations.get(columnIndex), column)
    }

    override fun rows(): Iterable<Group> {
        return query.groups as Iterable<Group>
    }

    override fun columns(): Iterable<String?> {
        return columnIndex.keys()
    }

    override fun getRowKey(index: Int): Group? {
        val groups: List<Group> = query!!.groups as List<Group>
        return if (index >= 0 && index < groups.size) {
            groups[index]
        } else {
            null
        }
    }

    override fun getColumnKey(index: Int): String {
        var index = index
        if (includeIndex) {
            if (index < query.dimensions!!.path.size) {
                return query.dimensions!!.path.get(index).toString()
            }
            index -= query.dimensions!!.path.size
        }
        return query.aggregations.get(index).name!!
    }

    override fun getRowAddress(row: Group?): Int {
        return rowIndex.getAddress(row!!)
    }

    override fun getColumnAddress(column: String?): Int {
        return columnIndex.getAddress(column!!)
    }

    override val rowCount: Int
        get() = query.groups!!.size
    override val columnCount: Int
        get() = if (includeIndex) {
            query.dimensions!!.path.size + query.aggregations.size
        } else {
            query.aggregations.size
        }

    fun reindexRowsUsingColumns(vararg rows: String?): DataFrame<MultiKey, String?, Any> {
        return ReIndexedDataFrame<MultiKey, String?, Any, Group, String>(this as DataFrame<Group, String, Any>, object : ReIndexRecipe<MultiKey, String?> {
            override fun buildRowIndex(): UniqueIndex<MultiKey> {
                var r = 0
                val keys: Array<MultiKey?> = arrayOfNulls<MultiKey>(rowCount)
                for (row in rows()) {
                    val values = arrayOfNulls<Any>(rows.size)
                    var c = 0
                    for (column in rows) {
                        values[c++] = getValueAt(row, column)
                    }
                    val key = MultiKey(values)
                    keys[r++] = key
                }
                return DefaultUniqueIndex<MultiKey>(*keys.requireNoNulls())
            }

            override fun buildColumnIndex(): UniqueIndex<String?> {
                return columnIndex
            }
        })
    }

    fun join(series: Series<*, *>?, columns: Array<String?>?): DataFrame<*, *, *>? {
        return null
    }

    private inner class DimensionSeries(dataFrame: DataFrame<Group, String, Any>, column: String, columnIndex: Int) :
        AbstractSeries<Group?, Any?>() {
        private val dataFrame: DataFrame<Group, String, Any>
        private val column: String
        private val columnIndex: Int
        override val name: Any
            get() = column
        override val type: KClass<out Any>
            get() = dataFrame.getColumnClass(column)

        override operator fun get(key: Group?): Any? {
            return if (key != null) {
                key.getPath(columnIndex)
            } else {
                null
            }
        }

        override fun getKey(i: Int): Group? {
            return getRowKey(i)
        }

        override fun size(): Int {
            return rowCount
        }

        override fun getAddress(key: Group?): Int {
            return getRowAddress(key)
        }

        override fun keys(): Iterable<Group> {
            return rows()
        }

        fun <L> reindex(vararg keys: L): Series<L, Any>? {
            return null
        }

        fun head(count: Int): Series<Group, Any>? {
            return null
        }

        fun tail(count: Int): Series<Group, Any>? {
            return null
        }

        init {
            this.dataFrame = dataFrame
            this.column = column
            this.columnIndex = columnIndex
        }
    }

    private inner class AggregationSeries(aggregation: Aggregation<Any?>, column: String) : AbstractSeries<Group?, Any?>() {
        private val aggregation: Aggregation<Any?>
        private val column: String
        override val name: Any
            get() = column
        override val type: KClass<Any>
            get() = Double::class as KClass<Any>

        override operator fun get(key: Group?): Any? {
            return query.getValue(key, aggregation)
        }

        override fun getKey(i: Int): Group? {
            return getRowKey(i)
        }

        override fun size(): Int {
            return rowCount
        }

        override fun getAddress(key: Group?): Int {
            return getRowAddress(key)
        }

        override fun keys(): Iterable<Group> {
            return rows()
        }

        fun <L> reindex(vararg keys: L): Series<L, Any>? {
            return null
        }

        fun head(count: Int): Series<Group, Any>? {
            return null
        }

        fun tail(count: Int): Series<Group, Any>? {
            return null
        }

        init {
            this.aggregation = aggregation
            this.column = column
        }
    }

    init {
        this.query = query
        this.includeIndex = includeIndex
        query.addWeakQueryListener(queryListener)
    }
}