package org.molap.dataframe

import com.macrofocus.common.selection.Selection
import com.macrofocus.common.selection.SelectionEvent
import com.macrofocus.common.selection.SelectionListener
import org.molap.index.DefaultUniqueIndex
import org.molap.index.UniqueIndex
import kotlin.reflect.KClass

/*
 * A row filterable data frame. Only the selected rows will be made visible by this data frame.
 *
 * @param <R > the type of row keys
 * @param <C > the type of column keys
 * @param <V > the type of values
*/
class SelectionDataFrame<R, C, V>(
    private val dataFrame: AbstractDataFrame<R, C, V>,
    private val filter: Selection<R>
) :
    AbstractDataFrame<R, C, V>() {
    private val dataFrameListener: DataFrameListener<R, C> = object : DataFrameListener<R, C> {
        override fun dataFrameChanged(event: DataFrameEvent<R, C>) {
            cacheRowIndex = null
            notifyDataFrameChanged(DataFrameEvent(null, null, true))
        }
    }
    private val selectionListener: SelectionListener<R> = object : SelectionListener<R> {
        override fun selectionChanged(event: SelectionEvent<R>) {
            cacheRowIndex = null
            notifyDataFrameChanged(DataFrameEvent(null, null, true))
        }
    }

    init {
        dataFrame.addWeakDataFrameListener(dataFrameListener)
        filter.addWeakSelectionListener(selectionListener)
    }

    var cacheRowIndex: UniqueIndex<R>? = null

    override val rowIndex: UniqueIndex<R>
        get() = if (cacheRowIndex == null) {
            val rows: MutableList<R> = ArrayList<R>()
            for (row in dataFrame.rows()) {
                if (filter.isSelected(row)) {
                    rows.add(row)
                }
            }
            cacheRowIndex = DefaultUniqueIndex<R>(rows)
            cacheRowIndex!!
        } else {
            cacheRowIndex!!
        }

    override val columnIndex: UniqueIndex<C>
        get() = dataFrame.columnIndex

    override fun getRowClass(r: R): KClass<*>? {
        return dataFrame.getRowClass(r)
    }

    override fun getColumnClass(c: C): KClass<out Any> {
        return dataFrame.getColumnClass(c)
    }

    override fun getValueAt(r: R, c: C): V {
        return dataFrame.getValueAt(r, c)
    }

    override fun rows(): Iterable<R> {
        return rowIndex.keys()
    }

    override fun columns(): Iterable<C> {
        return dataFrame.columns()
    }

    override fun getRowKey(index: Int): R {
        return rowIndex.getKey(index)
    }

    override fun getColumnKey(index: Int): C {
        return dataFrame.getColumnKey(index)
    }

    override fun getRowAddress(r: R): Int {
        return rowIndex.getAddress(r)
    }

    override fun getColumnAddress(c: C): Int {
        return dataFrame.getColumnAddress(c)
    }

    override val rowCount: Int
        get() = rowIndex.size
    override val columnCount: Int
        get() = dataFrame.columnCount
}
