package org.molap.dataframe

import org.molap.convert.AutoConverter
import org.molap.convert.TypeConverter
import org.molap.importer.DataFrameReader
import org.molap.index.DefaultUniqueIndex
import org.molap.index.IntegerRangeUniqueIndex
import org.molap.index.UniqueIndex
import kotlin.reflect.KClass

class SimpleDataFrame : AbstractMutableDataFrame<Int, String, Any?> {
    //    private MutableSeries<Integer,Object>[] series;
    override val columnIndex: DefaultUniqueIndex<String>
    override val rowIndex: UniqueIndex<Int>

    private var objects: Array<Array<Any?>>
    private val classes: Array<KClass<*>?>

    constructor(reader: DataFrame<Int,String,Any?>, autoConverter: AutoConverter? = null) {
        val names = arrayOfNulls<String>(reader.columnCount)
        classes = arrayOfNulls<KClass<*>>(reader.columnCount)

        for (i in 0..<reader.columnCount) {
            val columnName: Any? = reader.getColumnName(reader.getColumnKey(i))
            if (columnName != null) {
                names[i] = columnName.toString()
            } else {
                names[i] = null
            }
            classes[i] = reader.getColumnClass(reader.getColumnKey(i))
        }

        objects = Array(reader.rowCount) { arrayOfNulls(reader.columnCount) }
        for (row in 0..<reader.rowCount) {
            for (column in 0..<reader.columnCount) {
                var o: Any? = reader.getValueAt(reader.getRowKey(row), reader.getColumnKey(column))
                if (row > 0) {
                    if (objects[row - 1][column] != null) {
                        if (objects[row - 1][column] == o) {
                            o = objects[row - 1][column]
                        }
                    }
                }
                objects[row][column] = o
            }
        }

        val columns = arrayOfNulls<String>(names.size)
        val unique: MutableSet<String> = HashSet<String>()
        for (i in names.indices) {
            if (names[i] != null) {
                var header: String = names[i]!!
                val c = 0
                while (unique.contains(header)) {
                    header = "$header-$c"
                }
                columns[i] = header
                unique.add(header)
            } else {
                columns[i] = i.toString()
            }
        }
        columnIndex = DefaultUniqueIndex<String>(*columns.requireNoNulls())

        rowIndex = IntegerRangeUniqueIndex(0, objects.size - 1)

        if (autoConverter != null) {
            autoConvert(autoConverter)
        }
    }

    constructor(reader: DataFrameReader, autoConverter: AutoConverter? = null) {
        val names: Array<String?> = arrayOfNulls(reader.columnCount)
        classes = arrayOfNulls<KClass<*>>(reader.columnCount)
        for (i in 0..<reader.columnCount) {
            val columnName: Any? = reader.getColumnName(i)
            if (columnName != null) {
                names[i] = columnName.toString()
            } else {
                names[i] = null
            }
            classes[i] = reader.getClass(i)
        }
        var previousRow: Array<Any?>? = null
        if (reader.rowCount == -1) {
            val rows: MutableList<Array<Any?>> = ArrayList<Array<Any?>>()
            while (reader.hasMoreRow()) {
                reader.nextRow()
                if (reader.hasMoreColumn()) {
                    val row = arrayOfNulls<Any>(reader.columnCount)
                    var dataFound = false
                    for (i in 0..<reader.columnCount) {
                        val feature: Any?
                        if (reader.hasMoreColumn()) {
                            feature = reader.nextColumn()
                            if (feature != null) {
                                row[i] = feature
                                if (previousRow != null) {
                                    if (previousRow[i] != null) {
                                        if (previousRow[i] == row[i]) {
                                            row[i] = previousRow[i]
                                        }
                                    }
                                }
                                dataFound = true
                            }
                        } else {
                            row[i] = null
                        }
                    }
                    if (dataFound) {
                        rows.add(row)
                        previousRow = row
                    }
                }
            }
            objects = Array<Array<Any?>>(rows.size) { row ->
                rows[row]
            }
        } else {
            objects = Array(reader.rowCount) { arrayOfNulls(reader.columnCount) }
            for (row in 0..<reader.rowCount) {
                for (column in 0..<reader.columnCount) {
                    var o: Any? = reader.nextColumn()
                    if (row > 0) {
                        if (objects[row - 1][column] != null) {
                            if (objects[row - 1][column] == o) {
                                o = objects[row - 1][column]
                            }
                        }
                    }
                    objects[row][column] = o
                }
                reader.nextRow()
            }
        }

        val columns = arrayOfNulls<String>(names.size)
        val unique: MutableSet<String> = HashSet<String>()
        for (i in names.indices) {
            if (names[i] != null) {
                var header: String = names[i]!!
                val c = 0
                while (unique.contains(header)) {
                    header = "$header-$c"
                }
                columns[i] = header
                unique.add(header)
            } else {
                columns[i] = i.toString()
            }
        }
        columnIndex = DefaultUniqueIndex<String>(*columns.requireNoNulls())

        rowIndex = IntegerRangeUniqueIndex(0, objects.size - 1)

        if (autoConverter != null && !reader.isHasClassHeader) {
            autoConvert(autoConverter)
        }
    }

    override fun getRowClass(integer: Int): KClass<*> {
        return Any::class
    }

    override fun getColumnClass(s: String): KClass<*> {
        return classes[columnIndex.getAddress(s)]!!
    }

    override fun getValueAt(row: Int, column: String): Any? {
        return objects[row][columnIndex.getAddress(column)]
    }

    override fun setValueAt(row: Int, column: String, value: Any?) {
        if (objects[row][columnIndex.getAddress(column)] !== value) {
            objects[row][columnIndex.getAddress(column)] = value

            notifyDataFrameChanged(DataFrameEvent<Int, String>(null, null, true))
        }
    }

    private fun autoConvert(autoConverter: AutoConverter) {
        var fireEvent = false
        for (column in 0..<columnCount) {
            val c = column
            val input: TypeConverter.Input = object : TypeConverter.Input {
                override fun getType(): KClass<*> {
                    return getColumnClass(getColumnKey(c))
                }

                override fun get(index: Int): Any? {
                    return getValueAt(index, getColumnKey(c))
                }

                override fun size(): Int {
                    return rowCount
                }
            }
            val output: TypeConverter.Output = object : TypeConverter.Output {
                override fun setType(type: KClass<*>?) {
                    classes[c] = type
                }

                override fun set(index: Int, value: Any?) {
                    objects[index][c] = value
                }
            }

            fireEvent = fireEvent or autoConverter.convert(input, output)
        }
        if (fireEvent) {
            notifyDataFrameChanged(DataFrameEvent<Int, String>(null, null, true))
        }
    }
}
