package org.molap.datetime

import kotlinx.datetime.*
import kotlin.time.Duration

fun LocalDate.copy(
    year: Int? = null,
    monthNumber: Int? = null,
    dayOfMonth: Int? = null,
) = LocalDate(
    year ?: this.year,
    monthNumber ?: this.monthNumber,
    dayOfMonth ?: this.dayOfMonth,
)

fun LocalDateTime.copy(
    year: Int? = null,
    monthNumber: Int? = null,
    dayOfMonth: Int? = null,
    hour: Int? = null,
    minute: Int? = null,
    second: Int? = null,
    nanosecond: Int? = null,
) = LocalDateTime(
    year ?: this.year,
    monthNumber ?: this.monthNumber,
    dayOfMonth ?: this.dayOfMonth,
    hour ?: this.hour,
    minute ?: this.minute,
    second ?: this.second,
    nanosecond ?: this.nanosecond,
)

fun LocalDateTime.beginningOfHour() : LocalDateTime {
    return this.copy(minute = 0, second = 0, nanosecond = 0)
}

fun LocalDateTime.beginningOfMonth() : LocalDateTime {
    return this.date.copy(dayOfMonth = 1).atTime(0, 0)
}

operator fun LocalDateTime.minus(duration: Duration): LocalDateTime {
    return this.toInstant(TimeZone.UTC).minus(duration).toLocalDateTime(TimeZone.UTC)
}

operator fun LocalDateTime.minus(period: DatePeriod): LocalDateTime {
    return this.date.minus(period).atTime(this.hour, this.minute, this.second, this.nanosecond)
}

operator fun LocalDateTime.plus(duration: Duration): LocalDateTime {
    return this.toInstant(TimeZone.UTC).plus(duration).toLocalDateTime(TimeZone.UTC)
}

operator fun LocalDateTime.plus(period: DatePeriod): LocalDateTime {
    return this.date.plus(period).atTime(this.hour, this.minute, this.second, this.nanosecond)
}

fun LocalDateTime.startOf5MinutesInterval(): LocalDateTime {
    val truncatedMinutes = (minute / 5) * 5 // Calculate the start of the 5-minute interval
    return copy(
        minute = truncatedMinutes,
        second = 0,
        nanosecond = 0
    )
}

fun LocalDateTime.endOf5MinutesInterval(): LocalDateTime {
    val truncatedMinutes = ((minute / 5) + 1) * 5 // Move to the next 5-minute interval
    val adjustedHour = if (truncatedMinutes >= 60) hour + 1 else hour
    val adjustedMinutes = truncatedMinutes % 60

    // Handle day rollover
    val newDateTime = date.atTime((adjustedHour % 24), adjustedMinutes, 0, 0)

    return if (adjustedHour >= 24) newDateTime.plus(DatePeriod(days = 1)) else newDateTime
}