package org.molap.datetime

import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.datetime.*
import kotlin.time.Duration
import kotlin.time.Duration.Companion.hours
import kotlin.time.Duration.Companion.minutes

fun datetimeFlow(
    now: LocalDateTime = Clock.System.now().toLocalDateTime(TimeZone.UTC),
    start: LocalDate = now.date.minus(DatePeriod(years = 2)).copy(dayOfMonth = 1)
): Flow<LocalDateTime> = flow {
//    val startDate = now.date.minus(DatePeriod(years = 2)).copy(dayOfMonth = 1)

    // Emit the first day of each month for the past two years
    generateSequence(start) { it.plus(DatePeriod(months = 1)) }
        .takeWhile { it <= now.date.copy(dayOfMonth = 1) }
        .forEach { emit(it.atTime(0, 0)) }

    // Emit the current time every hour, starting from now
    while (true) {
        val currentTime = Clock.System.now().toLocalDateTime(TimeZone.currentSystemDefault())
        val nextHour = currentTime.copy(
            minute = 0,
            second = 0,
            nanosecond = 0
        ).toInstant(TimeZone.UTC).plus(1.hours)

        val delayMillis = nextHour
            .minus(Clock.System.now()).inWholeMilliseconds
        delay(delayMillis)
        emit(nextHour.toLocalDateTime(TimeZone.UTC))
    }
}

fun minutesFlow(
    start: LocalDateTime = Clock.System.now().toLocalDateTime(TimeZone.UTC),
    duration: Duration = 10.minutes,
    timeZone: TimeZone = TimeZone.UTC
): Flow<Pair<LocalDateTime,LocalDateTime>> = flow {
    // Emit the current time every hour, starting from now
    var currentTime = start

    while (true) {
        val nextHour = currentTime.copy(
            second = 0,
            nanosecond = 0
        ).toInstant(timeZone).plus(duration)

        val delayMillis = nextHour
            .minus(Clock.System.now()).inWholeMilliseconds
        delay(delayMillis)
        emit(Pair(start, nextHour.toLocalDateTime(timeZone)))

        currentTime = nextHour.toLocalDateTime(timeZone)
    }
}

/**
 * Emit the current time every hour, starting from now
 */
fun hourlyFlow(
    start: LocalDateTime = Clock.System.now().toLocalDateTime(TimeZone.UTC),
    end : LocalDateTime? = null,
    lag: Long = 0,
    timeZone: TimeZone = TimeZone.UTC
): Flow<Pair<LocalDateTime,LocalDateTime>> = flow {
    val duration: Duration = 1.hours

    var currentTime = start

    while (if(end != null) if(end > start) end >= currentTime else end < currentTime else true) {
        val nextHour: LocalDateTime
        if(end == null || end > start) {
            nextHour = currentTime.beginningOfHour().plus(duration)
        } else {
            nextHour = currentTime.beginningOfHour().minus(duration)
        }

        val delayMillis = nextHour.toInstant(TimeZone.UTC)
            .minus(Clock.System.now()).inWholeMilliseconds
        delay(delayMillis + lag)
        if(end == null || end > start) {
            emit(Pair(currentTime, nextHour))
        } else {
            emit(Pair(nextHour, currentTime))
        }

        currentTime = nextHour
    }
}

/**
 * Emit the first day of each month for the past two years
 */
fun monthlyFlow(
    start: LocalDateTime = Clock.System.now().toLocalDateTime(TimeZone.UTC),
    end: LocalDateTime = start.date.minus(DatePeriod(years = 2)).copy(dayOfMonth = 1).atTime(0, 0)
): Flow<Pair<LocalDateTime, LocalDateTime>> = flow {
    if(start < end ) {
        // From the past to now
        throw UnsupportedOperationException("From the past to now is not supported yet")
        generateSequence(end.date) { it.plus(DatePeriod(months = 1)) }
            .takeWhile { it <= start.date.copy(dayOfMonth = 1) }
            .forEach { emit(Pair(it.plus(DatePeriod(months = 1)).atTime(0, 0), it.atTime(0, 0))) }
    } else {
        val beginningOfMonth = start.beginningOfMonth()
        if(beginningOfMonth < start) {
            emit(Pair(beginningOfMonth, start))
        }

        val lastMonth = beginningOfMonth.minus(DatePeriod(months = 1))
        // Going back to history
        generateSequence(lastMonth) { it.minus(DatePeriod(months = 1)) }
            .takeWhile { it >= end }
            .forEach { emit(Pair(it, it.plus(DatePeriod(months = 1)))) }
    }
}
