/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.subset

import com.macrofocus.common.annotation.Synchronized
import com.macrofocus.common.collection.TIntArrayList
import com.macrofocus.common.collection.TIntList
import com.macrofocus.common.collection.asRandomAccessIterable
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.selection.SelectionEvent
import com.macrofocus.common.selection.SelectionListener

/**
 * Created by luc on 16/05/15.
 */
open class DefaultBinningDimension<Row, Column, Value, Bin>(
    dataFrame: SubsetDataFrame<Row, Column, Value>,
    private val column: Column,
    filteringCallback: FilteringCallback<Row>,
    selection: MutableSelection<Row>?,
    binningStrategy: SingleBinningDimension.SingleBinningStrategy<Row, Bin>?
) : AbstractBinningDimension<Row, Column, Value, Bin>(dataFrame, filteringCallback, selection, Operation.Or),
    CategoricalDimension<Row, Column, Bin> {

    private val selectionListener: SelectionListener<Row> = object : SelectionListener<Row> {
        override fun selectionChanged(event: SelectionEvent<Row>) {
            if (binningStrategy != null) {
                for (row in event.affected) {
                    if (binningStrategy.isBinnable(row)) {
                        val bin = rowToBin(row)
                        if (event.model.isSelected(row)) {
                            selectedCounts.adjustOrPutValue(bin, 1, 1)
                        } else {
                            selectedCounts.adjustValue(bin, -1)
                        }
                    }
                }
                notifySelectedCountChanged()
            }
        }
    }

    init {
        if (selection != null) {
            selection.addSelectionListener(selectionListener)
        }
    }


    override var binningStrategy: SingleBinningDimension.SingleBinningStrategy<Row, Bin>? = binningStrategy
        set(value) {
            if (field !== value) {
                updateFilterImmediately = true
                filterAll()
                field = value

                // Make sure the index exists
//            getIndex();
                markDirty()
                //            updateFilter();
                updateFilterImmediately = false
                // ToDo: Not clear what event should be sent
                notifyDimensionChanged(DimensionEvent<Row>(this, null, null))
                //            notifySelectedCountChanged();
            }
        }
    override val name: String?
        get() = column.toString()

    override fun containsBin(bin: Bin): Boolean {
        return index!!.containsKey(bin)
    }

    override fun getBin(row: Row): Bin {
        return rowToBin(row)
    }

    override fun getBins(row: Row): Iterable<Bin> {
        return listOf(rowToBin(row))
    }

    protected fun rowToBin(row: Row): Bin {
//        return if (binningStrategy != null) {
        return binningStrategy!!.rowToBin(row)
//        } else {
//             ToDO: Probably a bad design to return null in this case
//            null
//        }
    }

    @Synchronized
    override fun reduce(otherActiveIndices: IntArray?) {
        // If the binning strategy has not yet been defined (for example if no clustering has been performed but the dimension has already been created, then do nothing
        if (binningStrategy != null) {
            val oldIndices: IntArray? = otherActiveIndices
            this.otherActiveIndices = otherActiveIndices

            //        System.err.println("Reduce " + column + ", oldIndices=" + Arrays.toString(oldIndices) + ", newIndices=" + Arrays.toString(newIndices));
            if (otherActiveIndices != null) {
                val filtered: IntArray? = dataFrame.computeFiltered(oldIndices, otherActiveIndices)
                val unfiltered: IntArray? = dataFrame.computeUnfiltered(oldIndices, otherActiveIndices)
                println("Processing $filtered and $unfiltered with $binningStrategy")
                for (row in IndicesIterable<Row>(dataFrame.unfilteredDataFrame, filtered)) {
                    println("Processing $row with $binningStrategy")
                    if (binningStrategy!!.isBinnable(row)) {
                        val bin = rowToBin(row)
                        filterCounts!!.adjustOrPutValue(bin, 1, 1)
                        if (groups != null) {
                            for (group in groups!!.values) {
                                group.remove(bin, row)
                            }
                        }
//                        assert(filterCounts.get(bin) >= 0)
//                        assert(filterCounts.get(bin) <= getDensity(bin)) { "filterCount($bin) > getDensity($bin) for $name" }
                    }
                }
                for (row in IndicesIterable<Row>(dataFrame.unfilteredDataFrame, unfiltered)) {
                    if (binningStrategy!!.isBinnable(row)) {
                        val bin = rowToBin(row)
                        if (filterCounts!!.get(bin) > 1) {
                            filterCounts!!.adjustValue(bin, -1)
                        } else {
                            filterCounts!!.remove(bin)
                        }
                        if (groups != null) {
                            for (group in groups!!.values) {
                                group.add(bin, row)
                            }
                        }
//                        assert(filterCounts.get(bin) >= 0)
//                        assert(filterCounts.get(bin) <= getDensity(bin)) { "filterCount($bin) > getDensity($bin) for $name" }
                    }
                }
                updateActive()
                notifyDimensionChanged(
                    DimensionEvent<Row>(
                        this,
                        IndicesIterable(dataFrame.unfilteredDataFrame, filtered),
                        IndicesIterable(dataFrame.unfilteredDataFrame, unfiltered)
                    )
                )
            } else {
                filterCounts!!.clear()
                if (groups != null) {
                    for (group in groups!!.values) {
                        group.reset()
                    }
                }
                updateActive()
                notifyDimensionChanged(
                    DimensionEvent<Row>(
                        this,
                        IndicesIterable(dataFrame.unfilteredDataFrame, object : IndicesSupplier {
                            override fun get(): IntArray? {
                                return dataFrame.computeFiltered(oldIndices, otherActiveIndices)
                            }
                        }),
                        IndicesIterable(dataFrame.unfilteredDataFrame, object : IndicesSupplier {
                            override fun get(): IntArray? {
                                return dataFrame.computeUnfiltered(oldIndices, otherActiveIndices)
                            }
                        })
                    )
                )
            }
        }
    }

    override fun getGroup(reducer: Reducer<Row, Bin>): Group<Row, Bin> {
        if (groups == null) {
            groups = HashMap<Reducer<Row, Bin>, Group<Row, Bin>>()
        } else {
            if (groups!!.containsKey(reducer)) {
                return groups!!.get(reducer)!!
            }
        }
        val group: Group<Row, Bin> =
            DefaultGroup<Row, Bin>(bins!!.size(), reducer, dataFrame.unfilteredDataFrame, binningStrategy!!)
        groups!!.put(reducer, group)
        return group
    }

    override fun materializeIndex() {
        var index: MutableMap<Bin, TIntList> = HashMap<Bin, TIntList>()

        // ToDo: Series is probably unneeded, can obtain the row direction from dataFrame.unfilteredDataFrame
        if (binningStrategy != null) {
            this.selectedCounts.clear()
            val size: Int = dataFrame.unfilteredDataFrame.rowCount
            for (address in 0 until size) {
                val row: Row = dataFrame.unfilteredDataFrame.getRowKey(address)
                if (binningStrategy!!.isBinnable(row)) {
                    val bin = rowToBin(row)
                    val indices: TIntList?
                    if (index.containsKey(bin)) {
                        indices = index[bin]
                    } else {
                        indices = TIntArrayList()
                        index[bin] = indices
                    }
                    indices!!.add(address)

                    // Recompute how many items are selected in each bin
                    if (selection != null && selection!!.isSelected(row)) {
                        selectedCounts.adjustOrPutValue(bin, 1, 1)
                    }
                }
            }
        }
        val arrayIndex = createMap(index)
        var maxValueCount = 0
        var sumValueCount = 0
        if (binningStrategy != null) {
            for ((key, value) in index) {
                val indices: IntArray = value!!.toIntArray()
                arrayIndex[key] = indices
                if (indices.size > maxValueCount) {
                    maxValueCount = indices.size
                }
                sumValueCount += indices.size
            }
        }
        this.maxValueCount = maxValueCount
        this.sumValueCount = sumValueCount
        var bins: List<Bin> = ArrayList<Bin>(arrayIndex.keys)
        //        Collections.sort(bins, new UniversalComparator());
        this.index = arrayIndex
        this.bins = bins.asRandomAccessIterable()
        activeIndices = updateActiveIndices()
        updateActive()
    }

    protected fun createMap(map: Map<Bin, TIntList>): MutableMap<Bin, IntArray> {
        return HashMap<Bin, IntArray>(map.size)
    }

    override fun toString(): String {
        return "Dimension{" +
                "column=" + column +
                '}'
    }
}