/*
 * Copyright (c) 2015 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.subset

import com.macrofocus.common.collection.TIntHashSet
import com.macrofocus.common.collection.TIntSet
import com.macrofocus.common.command.Future
import com.macrofocus.common.concurrent.Callable
import com.macrofocus.common.concurrent.ExecutorService
import com.macrofocus.common.concurrent.Runtime
import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.filter.Filter
import com.macrofocus.common.filter.MutableIndexFilter
import com.macrofocus.common.filter.SimpleFilter
import com.macrofocus.common.filter.SimpleIndexFilter
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.timer.CPExecutor
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import org.molap.dataframe.DataFrame
import org.molap.dataframe.ReMappedDataFrame
import org.molap.dataframe.ReMappedRecipe
import org.molap.index.DefaultUniqueIndex
import org.molap.index.DefaultUniqueIndex.Companion.fromArray
import org.molap.index.UniqueIndex
import kotlin.jvm.JvmOverloads
import kotlin.math.max

/**
 * ToDo: Could use less memory by compressing all the pointers (int array), especially since direct traversal is not needed and accessing the pointers is unfrequent. Could be stored in a Gzipped byte array or ByteBuffer, or using some of the compression techniques of https://github.com/lemire/JavaFastPFOR
 *
 * ToDo: Also possible would to obtain how many unique values exist in each column to select appropriate data structure, as well as knowing whether the values could be accessed with less memory consuming (byte, short) pointers.
 *
 * @param <Row>
 * @param <Column>
 * @param <Value>
</Value></Column></Row> */
class SubsetDataFrame<Row, Column, Value>(dataFrame: DataFrame<Row, Column, Value>, outputFilter: MutableIndexFilter<Row>?, selection: MutableSelection<Row>?) :
    ReMappedDataFrame<Row, Column, Value>(dataFrame, DefaultReMappedRecipe<Row, Column, Value>(dataFrame)) {
    private var timer: CPTimer? = null
    private var executor: CPExecutor? = null
    private val outputFilter: MutableIndexFilter<Row>?
    private val selection: MutableSelection<Row>?
    private var activeIndices: IntArray? = null
    private val categoricalDimensions: MutableMap<Column, CategoricalDimension<Row, Column, Value>> = mutableMapOf()
    private val ordinalDimensions: MutableMap<Column, OrdinalDimension<Row, Column, Value>> = mutableMapOf()
    private val itemsDimension: ItemsDimension<Row, Column, Value>
    private val dimensions: MutableList<Dimension<Row>> = com.macrofocus.common.collection.CollectionFactory.copyOnWriteArrayList<Dimension<Row>>()
    private val filteringCallback: AbstractDimension.FilteringCallback<Row> = ReducingFilteringCallback<Any?>()

    @JvmOverloads
    constructor(dataFrame: DataFrame<Row, Column, Value>, outputFilter: MutableIndexFilter<Row>? = SimpleIndexFilter<Row>()) : this(
        dataFrame,
        outputFilter,
        null
    ) {
    }

    val unfilteredDataFrame: DataFrame<Row, Column, Value>
        get() = originalDataFrame

    fun getOutputFilter(): Filter<Row>? {
        return outputFilter
    }

    fun <Bin> getBinningDimension(
        column: Column,
        binningStrategy: SingleBinningDimension.SingleBinningStrategy<Row, Bin>?
    ): SingleBinningDimension<Row, Bin> {
        val dimension: SingleBinningDimension<Row, Bin> =
            DefaultBinningDimension<Row, Column, Value, Bin>(this, column, filteringCallback, selection, binningStrategy)
        dimensions.add(dimension)
        reduceDimension(dimension)
        return dimension
    }

//    fun <Bin> getBinningDimension(
//        binningStrategy: MultiBinningDimension.MultiBinningStrategy<Row, Bin>?,
//        vararg columns: Column
//    ): MultiBinningDimension<Row, Bin> {
//        val dimension: MultiBinningDimension<Row, Bin> =
//            DefaultMultiBinningDimension<Row, Column, Value, Bin>(this, filteringCallback, selection, binningStrategy, true, *columns)
//        dimensions.add(dimension)
//        reduceDimension(dimension)
//        return dimension
//    }

    fun getCategoricalDimension(column: Column): CategoricalDimension<Row, Column, Value> {
        return if (categoricalDimensions!!.containsKey(column)) {
            categoricalDimensions[column]!!
        } else {
            val dimension: CategoricalDimension<Row, Column, Value> =
                DefaultCategoricalDimension<Row, Column, Value>(this, column, filteringCallback, selection)
            dimensions.add(dimension)
            categoricalDimensions[column] = dimension
            reduceDimension(dimension)
            dimension
        }
    }

    fun <Bin> getDistributionDimension(
        column: Column,
        distributionStrategy: DistributionDimension.DistributionStrategy<Value, Bin>
    ): DistributionDimension<Row, Value, Bin> {
//        final UnivariateStatistics statistics = getStatistics(column);
//        new FixedBinningStrategy<Value>(10, statistics.getMinimum().doubleValue(), statistics.getMaximum().doubleValue();
        val dimension: DistributionDimension<Row, Value, Bin> =
            DefaultDistributionDimension<Row, Column, Value, Bin>(this, column, filteringCallback, selection, distributionStrategy)
        dimensions.add(dimension)
        reduceDimensions(dimension)
        return dimension
    }

    fun getOrdinalDimension(column: Column): OrdinalDimension<Row, Column, Value> {
        return if (ordinalDimensions.containsKey(column)) {
            ordinalDimensions[column]!!
        } else {
            val dimension: OrdinalDimension<Row, Column, Value> =
                DefaultOrdinalDimension<Row, Column, Value>(this, column, filteringCallback)
            dimensions.add(dimension)
            ordinalDimensions[column] = dimension
            reduceDimension(dimension)
            dimension
        }
    }

    fun getItemsDimension(): ItemsDimension<Row, Column, Value> {
        return itemsDimension
    }

//    fun createTextDimension(vararg columns: Column): DefaultTextDimension<Row, Column, Value> {
//        val dimension: DefaultTextDimension<Row, Column, Value> =
//            DefaultTextDimension<Row, Column, Value>(this, filteringCallback, selection, *columns)
//        addDimension(dimension)
//        return dimension
//    }

    fun addDimension(dimension: Dimension<Row>) {
        dimensions.add(dimension)
    }

    fun removeDimension(dimension: Dimension<Row>) {
        dimensions.remove(dimension)
        if (categoricalDimensions!!.containsValue(dimension)) {
            throw UnsupportedOperationException()
        }
        if (ordinalDimensions!!.containsValue(dimension)) {
            throw UnsupportedOperationException()
        }
    }

    /**
     * Filter using TIntList
     */
    fun updateFilters() {
        val oldIndices = activeIndices
        val newIndices = updateActiveIndices()
        val rowIndex: UniqueIndex<Row>
        rowIndex = if (newIndices != null) {
            originalDataFrame.rowIndex.keepAddresses(newIndices)
        } else {
            originalDataFrame.rowIndex
        }
        if (outputFilter != null && outputFilter.isEnabled) {
//            if(newIndices == null) {
//                outputFilter.clearFilter(this);
//            } else {
//                outputFilter.setFiltered(new AbstractDimension.IndicesIterable<Row>(unfilteredDataFrame, new AbstractDimension.IndicesSupplier() {
//                    @Override
//                    public int[] get() {
//                        return SubsetDataFrame.this.computeFiltered(oldIndices, newIndices);
//                    }
//                }), new AbstractDimension.IndicesIterable<Row>(unfilteredDataFrame, new AbstractDimension.IndicesSupplier() {
//                    @Override
//                    public int[] get() {
//                        return SubsetDataFrame.this.computeUnfiltered(oldIndices, newIndices);
//                    }
//                }), this);
//            }
            val filtered: AbstractDimension.IndicesSupplier = FilteredIndicesSupplier(oldIndices, newIndices)
            val unfiltered: AbstractDimension.IndicesSupplier = UnfilteredIndicesSupplier(oldIndices, newIndices)
            outputFilter.setIndex(
                rowIndex,
                AbstractDimension.IndicesIterable<Row>(unfilteredDataFrame, filtered),
                AbstractDimension.IndicesIterable<Row>(
                    unfilteredDataFrame, unfiltered
                ),
                if (newIndices == null) 0 else unfilteredDataFrame.rowCount - newIndices.size
            )
        }
        setRecipe(RowsReMappedRecipe(rowIndex))
    }

    private fun computeRowIndex(): UniqueIndex<Row> {
        return if (categoricalDimensions != null && categoricalDimensions.size > 0) {
            var intersection: IntArray? = null
            var noActiveIndices = true
            for (dimension in categoricalDimensions.values) {
                val activeIndices: IntArray? = dimension.activeIndices
                if (activeIndices != null) {
                    noActiveIndices = false
                    intersection = if (intersection == null) {
                        activeIndices
                    } else {
                        SortedArraySetOperations.intersectSortedArrays(intersection, activeIndices)
                    }
                }
            }
            val indices = intersection
            if (noActiveIndices) {
                originalDataFrame.rowIndex
            } else {
                if (indices != null) {
                    val rows = arrayOfNulls<Any>(indices.size) as Array<Row>
                    for (i in indices.indices) {
                        val index = indices[i]
                        rows[i] = originalDataFrame.getRowKey(index)
                    }
                    fromArray(rows)
                } else {
                    DefaultUniqueIndex<Row>()
                }
            }
        } else {
            originalDataFrame.rowIndex
        }
    }

    /**
     * Filter using raw arrays
     */
    fun updateFiltersUsingArrays() {
        setRecipe(object : ReMappedRecipe<Row, Column> {
            override fun buildRowIndex(): UniqueIndex<Row> {
                return if (categoricalDimensions != null && categoricalDimensions.size > 0 || ordinalDimensions != null && ordinalDimensions.size > 0) {
                    val intersection = intersectActiveIndices(null)
                    if (intersection != null) {
                        val rows = arrayOfNulls<Any>(intersection.size) as Array<Row>
                        for (i in intersection.indices) {
                            val index = intersection[i]
                            rows[i] = originalDataFrame.getRowKey(index)
                        }
                        fromArray(rows)
                    } else {
                        DefaultUniqueIndex<Row>()
                    }
                } else {
                    originalDataFrame.rowIndex
                }
            }

            override fun buildColumnIndex(): UniqueIndex<Column> {
                return originalDataFrame.columnIndex
            }
        })
    }

    fun updateActiveIndices(): IntArray? {
        val intersection = intersectActiveIndices(null)
        activeIndices = intersection
        return intersection
    }

    private fun intersectActiveIndices(skip: Dimension<*>?): IntArray? {
        var intersection: IntArray? = null
        for (dimension in dimensions) {
            if (dimension !== skip) {
                val activeIndices: IntArray? = dimension.activeIndices
                if (activeIndices != null) {
                    intersection = if (intersection == null) {
                        activeIndices
                    } else {
                        SortedArraySetOperations.intersectSortedArrays(intersection, activeIndices)
                    }
                }
            }
        }
        return intersection
    }

    private fun intersectActiveIndicesOld(skip: Dimension<*>): IntArray? {
        var intersection: IntArray? = null
        for (dimension in categoricalDimensions!!.values) {
            if (dimension !== skip) {
                val activeIndices: IntArray? = dimension.activeIndicesUsingArrays
                if (activeIndices != null) {
                    intersection = if (intersection == null) {
                        activeIndices
                    } else {
                        SortedArraySetOperations.intersectSortedArrays(intersection, activeIndices)
                    }
                }
            }
        }
        return intersection
    }

    @Deprecated("")
    fun updateFiltersUsingTHashSet() {
        setRecipe(object : ReMappedRecipe<Row, Column> {
            override fun buildRowIndex(): UniqueIndex<Row> {
                return if (categoricalDimensions != null && categoricalDimensions.size > 0) {
                    var intersection: TIntSet? = null
                    for (dimension in categoricalDimensions.values) {
                        val activeIndices: TIntSet? = dimension.activeIndicesUsingHashSet
                        if (activeIndices != null) {
                            if (intersection == null) {
                                intersection = TIntHashSet(activeIndices)
                            } else {
                                intersection.retainAll(activeIndices)
                            }
                        }
                    }
                    val indices: IntArray = intersection!!.toIntArray()
                    val rows = arrayOfNulls<Any>(indices.size) as Array<Row>
                    for (i in indices.indices) {
                        val index = indices[i]
                        rows[i] = originalDataFrame.getRowKey(index)
                    }
                    fromArray(rows)
                } else {
                    originalDataFrame.rowIndex
                }
            }

            override fun buildColumnIndex(): UniqueIndex<Column> {
                return originalDataFrame.columnIndex
            }
        })
    }

    fun defineIndex(vararg columns: Column) {
        for (column in columns) {
            getCategoricalDimension(column)
        }
    }

    fun materializeIndex(vararg columns: Column) {
        val todo: MutableList<Callable<Column>> = ArrayList<Callable<Column>>()
        for (column in columns) {
            todo.add(object : Callable<Column> {
                override fun call(): Column {
                    val dimension: Dimension<*> = getCategoricalDimension(column)
                    dimension.materializeIndex()
                    return column
                }
            })
        }
        if (MULTITHREADED) {
            val executor: ExecutorService? = CPHelper.instance.newFixedThreadPool(SubsetDataFrame::class.simpleName!!, nAvailableProcessors / 2, nAvailableProcessors)
            try {
                val answers: List<Future<Column>> = executor!!.invokeAll(todo)
                for (answer in answers) {
                    try {
                        answer.get()
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
            } catch (e: Exception) {
                e.printStackTrace() // something wrong, maybe
            } finally {
                executor?.shutdown()
            }
        } else {
            for (callable in todo) {
                try {
                    callable.call()
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
        }
    }

    protected override fun originalDataFrameChanged() {
        invalidateFilters()
        scheduleUpdateFilters()
    }

    private val timerListener: CPTimerListener = object : CPTimerListener {
        override fun timerTriggered() {
            updateFilters()
        }
    }

    fun setTimer(timer: CPTimer?) {
        this.timer?.removeTimerListener(timerListener)
        this.timer = timer
        timer?.addTimerListener(timerListener)
    }

    protected fun invalidateFilters() {
        for (dimension in dimensions) {
            dimension.markDirty()
        }
        activeIndices = null
    }

    protected fun scheduleUpdateFilters() {
        if (timer != null) {
            timer!!.restart()
        } else {
            updateFilters()
        }
    }

    fun computeFiltered(oldIndices: IntArray?, newIndices: IntArray?): IntArray? {
        val filtered: IntArray?
        filtered = if (oldIndices == null) {
            if (newIndices == null) {
                null
            } else {
                // Everything was active
                SortedArraySetOperations.diffSortedTIntArrays(unfilteredDataFrame.rowCount, newIndices)
            }
        } else {
            if (newIndices == null) {
                // Everything is now active
                null
            } else {
                SortedArraySetOperations.diffSortedTIntList(oldIndices, newIndices)
            }
        }
        return filtered
    }

    fun computeUnfiltered(oldIndices: IntArray?, newIndices: IntArray?): IntArray? {
        val unfiltered: IntArray?
        unfiltered = if (oldIndices == null) {
            if (newIndices == null) {
                null
            } else {
                // Everything was active
                null
            }
        } else {
            if (newIndices == null) {
                // Everything is now active
                SortedArraySetOperations.diffSortedTIntArrays(unfilteredDataFrame.rowCount, oldIndices)
            } else {
                SortedArraySetOperations.diffSortedTIntList(newIndices, oldIndices)
            }
        }
        return unfiltered
    }

    fun reset() {
        for (dimension in dimensions) {
            dimension.filterAll()
        }
    }

    fun setExecutor(executor: CPExecutor?) {
        this.executor = executor
    }

    private inner class ReducingFilteringCallback<Bin> : AbstractDimension.FilteringCallback<Row> {
        override fun filteringChanged(event: AbstractDimension.FilteringEvent<Row>) {
            val dimensionToSkip: Dimension<Row> = event.getDimension()
            reduceDimensions(dimensionToSkip)
        }
    }

    private var cancelableReduceDimensions: CPExecutor.Cancelable? = null
    private fun reduceDimensions(dimensionToSkip: Dimension<Row>) {
        val commands: MutableList<CPExecutor.Command> = ArrayList<CPExecutor.Command>()
        for (dimension in dimensions) {
            if (dimension.isReducable && dimension !== dimensionToSkip) {
                commands.add(object : CPExecutor.Command {
                    override fun execute(): Boolean {
                        reduceDimension(dimension)
                        return false
                    }
                })
            }
        }
        if (commands.size > 0) {
            val command: CPExecutor.Command = object : CPExecutor.Command {
                var iterator: Iterator<CPExecutor.Command> = commands.iterator()
                override fun execute(): Boolean {
                    iterator.next().execute()
                    return iterator.hasNext()
                }
            }
            if (executor != null) {
                if (cancelableReduceDimensions != null) {
                    cancelableReduceDimensions!!.cancel()
                }
                cancelableReduceDimensions = executor!!.scheduleIncremental(command)
            } else {
                while (command.execute());
            }
        }
        scheduleUpdateFilters()
    }

    fun reduceDimension(dimension: Dimension<Row>) {
        val otherActiveIndices = intersectActiveIndices(dimension)

//                    System.err.println(event.getDimension() + "->" + dimension + ", filtered=" + event.getFiltered() + ", unfiltered" + event.getUnfiltered() + ", " + ", otherActiveIndices=" + Arrays.toString(otherActiveIndices));
        dimension.reduce(otherActiveIndices)
    }

    private inner class FilteredIndicesSupplier(private val oldIndices: IntArray?, private val newIndices: IntArray?) :
        AbstractDimension.IndicesSupplier {
        override fun get(): IntArray? {
            return computeFiltered(oldIndices, newIndices)
        }
    }

    private inner class UnfilteredIndicesSupplier(private val oldIndices: IntArray?, private val newIndices: IntArray?) :
        AbstractDimension.IndicesSupplier {
        override fun get(): IntArray? {
            return computeUnfiltered(oldIndices, newIndices)
        }
    }

    private inner class RowsReMappedRecipe(rowIndex: UniqueIndex<Row>) : ReMappedRecipe<Row, Column> {
        private val rowIndex: UniqueIndex<Row>
        override fun buildRowIndex(): UniqueIndex<Row> {
            return rowIndex
        }

        override fun buildColumnIndex(): UniqueIndex<Column> {
            return originalDataFrame.columnIndex
        }

        init {
            this.rowIndex = rowIndex
        }
    }

    companion object {
        private const val MULTITHREADED = false
        private val nAvailableProcessors: Int = max(Runtime.getRuntime().availableProcessors().toInt(), 1.toInt())
    }

    init {
        itemsDimension = DefaultItemsDimension<Row, Column, Value>(
            this,
            if (outputFilter != null) outputFilter.inputFilter!! else SimpleFilter<Row>(),
            filteringCallback
        )
        dimensions.add(itemsDimension)
        this.outputFilter = outputFilter
        this.selection = selection
    }
}