@file:OptIn(ExperimentalTime::class)

package org.molap.aggregates.aggregation

import kotlinx.datetime.*
import org.molap.aggregates.cube.Group
import org.molap.aggregates.cuboid.Cuboid
import org.molap.index.DefaultUniqueIndex
import org.molap.index.UniqueIndex
import org.molap.series.AbstractTimeSeries
import org.molap.series.Series
import org.molap.series.TimeSeries
import kotlin.reflect.KClass
import kotlin.time.ExperimentalTime
import kotlin.time.Instant

class TimeSeriesAggregation(time: Series<*,*>, column: Series<*,*>) : AbstractAggregation<TimeSeries<*,*>?>() {
    private val time: Series<*,*> = time
    private val column: Series<*,*> = column

    private val func: AggregateSeriesFunction<TimeSeries<*,*>> =
        TimeSeriesAggregateSeriesFunction(time, column)

    override val type: KClass<Any>
        get() = TimeSeries::class as KClass<Any>

    override fun compute(cuboid: Cuboid?, group: Group?): TimeSeries<*,*> {
        return cuboid!!.getAggregate(group, func)!!
    }

    override fun toString(): String {
  //      return "TimeSeries(" + column.name + ")"
        return column.name.toString()
    }

    private class TimeSeriesAggregateSeriesFunction(time: Series<*,*>, column: Series<*,*>) :
        AggregateSeriesFunction<TimeSeries<*,*>> {
        private val time: Series<*,*> = time
        private val column: Series<*,*> = column

        override fun aggregate(cuboid: Cuboid, rows: Iterable<Any?>?): TimeSeries<*,*> {
            return DataFrameTimeSeries<Any?, Any?, Any?>(time as Series<Any?,Any?>, column as Series<Any?,Any?>, rows!!)
        }

        override fun equals(o: Any?): Boolean {
            if (this === o) return true
            if (o == null || this::class != o::class) return false
            val that = o as TimeSeriesAggregateSeriesFunction
            return time == that.time &&
                    column == that.column
        }

        override fun hashCode(): Int {
            return hash(time, column)
        }

        fun hash(vararg values: Any?): Int {
            return values.fold(1) { acc, value -> 31 * acc + (value?.hashCode() ?: 0) }
        }
    }

    class DataFrameTimeSeries<R, C, V>(time: Series<R, V>, column: Series<R, V>, rows: Iterable<R>) :
        AbstractTimeSeries<R, V>(), TimeSeries<R, V> {
        private val time: Series<R, V> = time
        private val column: Series<R, V> = column
        private val rows: UniqueIndex<R>

        //            if(time.size() > 0) {
//                return getTime(getKey(0));
//            } else {
//                return 0;
//            }
        override var minTime: Long = 0
            private set

        //            if(time.size() > 0) {
//                return getTime(getKey(time.size() - 1));
//            } else {
//                return 0;
//            }
        override var maxTime: Long = 0
            private set

        init {
            this.rows = DefaultUniqueIndex(rows, true)

            for (row in rows) {
                val t = getAbsoluteTime(row)
                if (t > maxTime) {
                    maxTime = t
                }
                if (minTime == 0L || t < minTime) {
                    minTime = t
                }
            }
        }

        override fun getAbsoluteTime(key: R): Long {
            val v = time.get(key) as V
            return if(v is Instant) {
                v.toEpochMilliseconds()
            } else if (v is LocalDateTime) {
                v.toInstant(TimeZone.currentSystemDefault()).toEpochMilliseconds()
            } else if(v is LocalDate) {
                v.atStartOfDayIn(TimeZone.currentSystemDefault()).toEpochMilliseconds()
            } else {
                (v as Number).toLong()
            }
        }

        override fun getRelativeTime(key: R): Long {
            return getAbsoluteTime(key) - minTime
        }

        override fun getSequantialTime(key: R): Long {
            return rows.getAddress(key).toLong()
        }

        override val name: Any?
            get() = column.name

        override val type: KClass<out Any>
            get() = column.type

        override fun get(key: R): V {
            return column.get(key)
        }

        override fun getKey(i: Int): R {
            return rows.getKey(i)
        }

        override fun size(): Int {
            return rows.size
        }

        override fun getAddress(key: R): Int {
            return rows.getAddress(key)
        }

        override fun keys(): Iterable<R> {
            return rows.keys()
        }

        override fun toString(): String {
            return "DataFrameTimeSeries{" +
                    "time=" + time +
                    ", column=" + column +
                    '}'
        }
    }
}
