/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates.query

import com.macrofocus.common.collection.CollectionFactory.copyOnWriteArrayList
import com.macrofocus.common.collection.WeakReference
import org.molap.aggregates.aggregation.Aggregation
import org.molap.aggregates.cuboid.Cuboid
import org.molap.dataframe.DataFrame

abstract class AbstractQuery : Query {
    var isDirty = true
    private val listeners: MutableList<QueryListener>
    override fun drillDown(vararg columns: Any?): CuboidQuery {
        return CuboidQuery(this, CuboidQuery.DrillDownOperation(*columns), *aggregations)
    }

    override fun drillUp(): Query {
        return CuboidQuery(this, CuboidQuery.DrillUpOperation(), *aggregations)
    }

    override fun slice(value: Any?): Query {
        return CuboidQuery(this, CuboidQuery.SliceOperation(value), *aggregations)
    }

    override fun dice(valuesSets: Set<Any?>?): Query {
        return CuboidQuery(this, CuboidQuery.DiceOperation(valuesSets), *aggregations)
    }

    override fun pivot(aggregation: Aggregation<Any?>?, columns: Array<Any?>?): Query? {
        return PivotQuery(this, aggregation!!, columns)
    }

    override fun pivot(aggregation: Aggregation<Any?>?): Query {
        return PivotQuery(this, aggregation!!)
    }

    override fun collapse(): Query {
        return CuboidQuery(this, CuboidQuery.CollapseOperation(), *aggregations)
    }

    override fun order(vararg aggregations: Aggregation<Any?>?): Query {
        val a :  Array<Aggregation<Any?>> = aggregations as Array<Aggregation<Any?>>
        return OrderQuery(this, a)
    }

    override fun on(vararg cuboids: Cuboid): Query {
        return CuboidsQuery(cuboids, *aggregations)
    }

    override fun asDataFrame(includeIndex: Boolean): DataFrame<*,*,*> {
        return QueryDataFrame<Any?>(this, includeIndex)
    }

    override fun addQueryListener(listener: QueryListener) {
        listeners.add(listener)
    }

    override fun addWeakQueryListener(listener: QueryListener) {
        val weakListener: WeakQueryListener<*> = WeakQueryListener<Any?>(listener)
        listeners.add(weakListener)
    }

    override fun removeQueryListener(listener: QueryListener) {
        if (listener is WeakQueryListener<*>) {
            val removed = listeners.remove(listener)
//                assert(removed) { listener }
        } else {
            var toRemove: QueryListener? = null
            for (queryListener in listeners) {
                var comparable: QueryListener?
                if (queryListener is WeakQueryListener<*>) {
                    comparable = queryListener.reference
                } else {
                    comparable = queryListener
                }
                if (listener.equals(comparable)) {
                    toRemove = queryListener
                }
            }
            if (toRemove != null) {
                val removed = listeners.remove(toRemove)
//                    assert(removed) { listener }
            }
        }
    }

    override fun removeQueryListeners() {
        listeners.clear()
    }

    protected fun notifyQueryChanged() {
        for (listener in listeners) {
            listener.queryChanged()
        }
    }

    private inner class WeakQueryListener<E>(listener: QueryListener) : QueryListener {
        private val l_ref: WeakReference<QueryListener>
        override fun queryChanged() {
            val l: QueryListener? = reference
            if (l != null) {
                l.queryChanged()
            } else {
                removeQueryListener(this)
            }
        }

        val reference: QueryListener?
            get() = l_ref.get()

        override fun toString(): String {
            val l: QueryListener? = reference
            return if (l != null) {
                "Weak[$l]"
            } else {
                super.toString()
            }
        }

        init {
            l_ref = WeakReference<QueryListener>(listener)
        }
    }

    companion object {
        private fun abbreviate(str: String?, maxWidth: Int): String? {
            if (str == null) {
                return null
            }
            return if (str.length <= maxWidth) {
                str
            } else str.substring(0, maxWidth - 3) + "..."
        }
    }

    init {
        listeners = copyOnWriteArrayList<QueryListener>()
    }
}