/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates.query

import com.macrofocus.common.collection.arraycopy

class Structure protected constructor(vararg path: Boolean) {
    val path: BooleanArray
    fun drillUp(): Structure? {
        return if (path.size > 0) {
            getDrillUp(*path)
        } else {
            null
        }
    }

    fun drillDown(vararg columns: Boolean): Structure {
        val path = BooleanArray(path.size + columns.size)
        arraycopy(this.path, 0, path, 0, this.path.size)
        arraycopy(columns, 0, path, this.path.size, columns.size)
        return create(*path)
    }

    fun drillDown(valueAt: Boolean): Structure {
        return getDrillDown(valueAt, *path)
    }

    val last: Boolean
        get() = path[path.size - 1]

    fun changeLast(value: Boolean): Structure {
        return drillUp()!!.drillDown(value)
    }

    override fun equals(o: Any?): Boolean {
        if (this === o) return true
        if (o !is Structure) return false

        // Probably incorrect - comparing Boolean[] arrays with Arrays.equals
        return path.contentEquals(o.path)
    }

    override fun hashCode(): Int {
        return path.contentHashCode()
    }

    override fun toString(): String {
        return "Structure{" +
                "path=" + path.contentToString() +
                '}'
    }

    fun startsWith(structure: Structure?): Boolean {
        if (this === structure) return true
        if (structure == null) return false
        val length = path.size
        if (structure.path.size >= length) return false
        for (i in structure.path.indices) {
            val o1 = path[i]
            val o2 = structure.path[i]
            if (!(if (o1 == null) o2 == null else o1 == o2)) return false
        }
        return true
    }

    companion object {
        private operator fun get(vararg path: Boolean): Structure {
            return Structure(*path)
        }

        private fun getDrillDown(last: Boolean, vararg path: Boolean): Structure {
            val newPath = BooleanArray(path.size + 1)
            arraycopy(path, 0, newPath, 0, path.size)
            newPath[newPath.size - 1] = last
            return get(*newPath)
        }

        private fun getDrillUp(vararg path: Boolean): Structure {
            val newPath = BooleanArray(path.size - 1)
            arraycopy(path, 0, newPath, 0, newPath.size)
            return get(*newPath)
        }

        fun createRoot(): Structure {
            return get()
        }

        fun create(vararg path: Boolean): Structure {
            return get(*path)
        }
    }

    init {
        this.path = path
    }
}