package org.molap.dataframe

import org.molap.index.UniqueIndex
import kotlin.reflect.KClass

/**
 * A reindexed data frame.
 *
 * @param <R> the rexindexed type of rows
 * @param <C> the reindexed type of columns
 * @param <V> the type of values
</V></C></R> */
open class ReMappedDataFrame<R, C, V>(open var originalDataFrame: DataFrame<R, C, V>, recipe: ReMappedRecipe<R, C>?) :
    AbstractMutableDataFrame<R, C, V>(), MutableDataFrame<R, C, V> {
    var cacheColumnIndex: UniqueIndex<C>? = null
    var cacheRowIndex: UniqueIndex<R>? = null
    override val columnIndex: UniqueIndex<C>
        get() = if (cacheColumnIndex == null) {
            if(recipe != null) {
                cacheColumnIndex = recipe?.buildColumnIndex()
                cacheColumnIndex!!
            } else {
                originalDataFrame.columnIndex
            }
        } else {
            cacheColumnIndex!!
        }
    override val rowIndex: UniqueIndex<R>
        get() = if (cacheRowIndex == null) {
            if(recipe != null) {
                cacheRowIndex = recipe?.buildRowIndex()
                cacheRowIndex!!
            } else {
                originalDataFrame.rowIndex
            }
        } else {
            cacheRowIndex!!
        }
    private var recipe: ReMappedRecipe<R, C>?
    private val dataFrameListener: DataFrameListener<R, C> = object : DataFrameListener<R, C> {
        override fun dataFrameChanged(event: DataFrameEvent<R, C>) {
            originalDataFrameChanged()
        }
    }

    fun setDataFrame(dataFrame: DataFrame<R, C, V>) {
        if (originalDataFrame !== dataFrame) {
            if (originalDataFrame != null) {
                originalDataFrame.removeDataFrameListener(dataFrameListener)
            }
            originalDataFrame = dataFrame
            if (originalDataFrame != null) {
                originalDataFrame.addDataFrameListener(dataFrameListener)
            }
            invalidate()
            scheduleUpdate()
        }
    }

    fun setRecipe(recipe: ReMappedRecipe<R, C>) {
        if (this.recipe !== recipe) {
            this.recipe = recipe
            invalidate()
            scheduleUpdate()
        }
    }

    protected open fun originalDataFrameChanged() {
        invalidate()
        scheduleUpdate()
    }

    protected fun invalidate() {
        cacheRowIndex = null
        cacheColumnIndex = null
    }

    protected fun scheduleUpdate() {
        updateImmediatelly()
    }

    protected fun updateImmediatelly() {
        notifyDataFrameChanged(DataFrameEvent(emptyList(), emptyList(), true))
    }

    override fun getRowClass(row: R): KClass<*>? {
        return originalDataFrame.getRowClass(row)
    }

    override fun getColumnClass(column: C): KClass<*> {
        return originalDataFrame.getColumnClass(column)
    }

    override fun getValueAt(row: R, column: C): V {
//        assert(getColumnIndex().contains(column)) { "Column $column doesn't exists" }
        return originalDataFrame.getValueAt(row, column)
    }

    override fun setValueAt(row: R, column: C, value: V) {
        if (originalDataFrame is MutableDataFrame) {
            (originalDataFrame as MutableDataFrame<R, C, V>).setValueAt(row, column, value)
        }
    }

//    fun getRow(row: R): Series<C, *>? {
//        return null
//    }

//    fun join(series: Series?, columns: Array<C>?): DataFrame {
//        throw IllegalStateException("Not implemented")
//    }

//    fun getRowIndex(): UniqueIndex<R>? {
//        return if (recipe != null) {
//            if (rowIndex == null) {
//                rowIndex = recipe.buildRowIndex()
//            }
//            rowIndex
//        } else {
//            //            GWT.log("rowindex: " + rowIndex);
//            originalDataFrame.getRowIndex()
//        }
//    }
//
//    fun getColumnIndex(): UniqueIndex<C>? {
//        return if (recipe != null) {
//            if (columnIndex == null) {
//                columnIndex = recipe.buildColumnIndex()
//            }
//            columnIndex
//        } else {
//            originalDataFrame.getColumnIndex()
//        }
//    }

    init {
        this.recipe = recipe
        if (originalDataFrame != null) {
            originalDataFrame.addWeakDataFrameListener(dataFrameListener)
        }
    }
}
