@file:OptIn(ExperimentalTime::class)

package org.molap.datetime

import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlin.time.Clock
import kotlin.time.Instant
import kotlin.time.Duration
import kotlin.time.Duration.Companion.minutes
import kotlin.time.ExperimentalTime

/**
 * Throttle processing to one event per minute, accounting for processing time
 */
public suspend fun <T>  rateLimit(timeBetweenEvents: Duration = 1.minutes, function: suspend () -> T): T {
    val startTime: Instant = Clock.System.now() // Record the start time

    val t = function()

    // Calculate the remaining delay
    val elapsedTime = Clock.System.now() - startTime
    val remainingDelayMillis = timeBetweenEvents.inWholeMilliseconds - elapsedTime.inWholeMilliseconds

    // Delay for the remaining time if needed
    if (remainingDelayMillis > 0) {
        delay(remainingDelayMillis)
    }

    return t
}

public fun <T> Flow<T>.rateLimit(timeBetweenEvents: Duration = 1.minutes): Flow<T> = flow {
    collect { value ->
        val startTime: Instant = Clock.System.now() // Record the start time

        emit(value)

        // Calculate the remaining delay
        val elapsedTime = Clock.System.now() - startTime
        val remainingDelayMillis = timeBetweenEvents.inWholeMilliseconds - elapsedTime.inWholeMilliseconds

        // Delay for the remaining time if needed
        if (remainingDelayMillis > 0) {
            delay(remainingDelayMillis)
        }
    }
}
