/*
 * Copyright (c) 2009 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.data;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.net.URL;
import java.util.StringTokenizer;
import java.util.Vector;

public final class SimpleDataModel extends AbstractDataModel implements Externalizable {
    static final long serialVersionUID = -3175316423606085277L;

    private Object[][] objects;
    private Object[] columnNames;
    private Class[] classes;

    public SimpleDataModel() {
    }

    public SimpleDataModel(final DataModel dataModel) {
        columnNames = new Object[dataModel.getColumnCount()];
        classes = new Class[dataModel.getColumnCount()];
        objects = new Object[dataModel.getRowCount()][dataModel.getColumnCount()];
        for (int column = 0; column < dataModel.getColumnCount(); column++) {
            columnNames[column] = dataModel.getColumnName(column);
            classes[column] = dataModel.getColumnClass(column);
            for (int row = 0; row < dataModel.getRowCount(); row++) {
                if (dataModel.isAvailable(row, column)) {
                    objects[row][column] = dataModel.getValueAt(row, column);
                } else {
                    objects[row][column] = null;
                }
            }
        }
    }

    /**
     * Creates a new <code>SimpleDataModel</code> instance.
     *
     * @param reader a <code>java.io.Reader</code> value
     * @deprecated This method is no longer used. Its functionality has been replaced by <code>SimpleDataModel(final
     *             DataReader reader)</code>
     */
    public SimpleDataModel(Reader reader) {
        int dimCount = 0;
        String token = "";
        LineNumberReader lnr = new LineNumberReader(reader);
        java.lang.reflect.Constructor[] constructors;
        Vector array = new Vector();
        try {
            StringTokenizer header1 = new StringTokenizer(lnr.readLine(), "\t");
            StringTokenizer header2 = new StringTokenizer(lnr.readLine(), "\t");
            dimCount = header2.countTokens();
            columnNames = new Object[dimCount];
            classes = new Class[dimCount];
            constructors = new java.lang.reflect.Constructor[dimCount];
            Class[] cl = {String.class};
            for (int i = 0; i < dimCount; i++) {
                columnNames[i] = header1.nextToken();
                String c = header2.nextToken();
                try {
                    classes[i] = Class.forName(c);
                } catch (ClassNotFoundException e) {
                    classes[i] = Class.forName("java.lang." + c);
                }
                if (classes[i] == null) {
                    throw new ClassNotFoundException(c);
                }
                constructors[i] = classes[i].getConstructor(cl);
                if (constructors[i] == null) {
                    throw new ClassNotFoundException(cl.toString());
                }
            }
            while (lnr.ready()) {
                final String s = lnr.readLine();
                if (s == null) {
                    break;
                }
                final Vector vector = new Vector(dimCount);
                final StringTokenizer st = new StringTokenizer(s, "\t");
                for (int i = 0; i < dimCount; i++) {
                    Object feature;
                    if (st.hasMoreTokens()) {
                        token = st.nextToken().trim();
                    } else {
                        token = null;
                    }
                    if (token == null || token.equals("") || token.equals("n.a.") || token.equals("N/A") || token.equals("x") || token.equals("NaN") || token.equals("---") || token.equals("Infinity") || token.equals("-Infinity")) {
                        vector.addElement(null);
                    } else {
                        //                         try {
                        //                             java.lang.reflect.Field field = classes[i].getField(token);
                        //                             feature = field.get(null);
                        //                         }
                        //                         catch(NoSuchFieldException e) {
                        Object[] initargs = {token};
                        try {
                            feature = constructors[i].newInstance(initargs);


                            if (feature == null) {
                                System.err.println("Instantiation failed for " + token + " as " + constructors[i] + " at line " + lnr.getLineNumber());
                                throw new NoSuchFieldException("Instantiation failed for " + token + ",constructor=" + constructors[i]);
                            }
                            vector.addElement(feature);
                        } catch (Exception e) {
                            e.printStackTrace();
                            System.err.println("Instantiation failed for " + token + " as " + constructors[i] + " at line " + lnr.getLineNumber());
                        }
                    }
                }
                array.addElement(vector);
            }
            objects = new Object[array.size()][dimCount];
            for (int j = 0; j < dimCount; j++) {
                for (int i = 0; i < array.size(); i++) {
                    Object value = ((Vector) array.elementAt(i)).elementAt(j);
                    objects[i][j] = value;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("Error in reading line number " + lnr.getLineNumber() + ". Token is " + token);
        }

        try {
            reader.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    public Object getValueAt(int row, int column) {
        return objects[row][column];
    }

    @Override
    public boolean isAvailable(int row, int column) {
        return objects[row][column] != null;
    }

    @Override
    public Object getColumnName(int column) {
        return columnNames[column];
    }

    @Override
    public Class getColumnClass(int column) {
        return classes[column];
    }

    @Override
    public int getRowCount() {
        return objects.length;
    }

    @Override
    public int getColumnCount() {
        return columnNames.length;
    }

    @Override
    public void writeExternal(ObjectOutput out) throws IOException {
        out.writeInt(10000);
        out.writeInt(getRowCount());
        out.writeInt(getColumnCount());
        for (int column = 0; column < getColumnCount(); column++) {
            out.writeObject(columnNames[column]);
        }
        for (int column = 0; column < getColumnCount(); column++) {
            out.writeObject(classes[column]);
        }
        for (int row = 0; row < getRowCount(); row++) {
            for (int column = 0; column < getColumnCount(); column++) {
                out.writeBoolean(isAvailable(row, column));
            }
        }
        for (int column = 0; column < getColumnCount(); column++) {
            final Class cl = getColumnClass(column);
            if (cl == Double.class) {
                for (int row = 0; row < getRowCount(); row++) {
                    if (isAvailable(row, column)) {
                        if(objects[row][column] instanceof Number) {
                            out.writeDouble(((Number) objects[row][column]).doubleValue());
                        } else {
                            System.err.println("Row " + row + ", column " + column +
                                    ": " + objects[row][column].getClass().getSimpleName() + " (" + objects[row][column] +
                                    ") cannot be cast to Number");
                        }
                    }
                }
            } else if (cl == Float.class) {
                for (int row = 0; row < getRowCount(); row++) {
                    if (isAvailable(row, column)) {
                        if(objects[row][column] instanceof Float) {
                            out.writeFloat(((Float) objects[row][column]).floatValue());
                        } else {
                            System.err.println("Row " + row + ", column " + column +
                                    ": " + objects[row][column].getClass().getSimpleName() + " (" + objects[row][column] +
                                    ") cannot be cast to Float");
                        }
                    }
                }
            } else if (cl == Long.class) {
                for (int row = 0; row < getRowCount(); row++) {
                    if (isAvailable(row, column)) {
                        if(objects[row][column] instanceof Long) {
                            out.writeLong(((Long) objects[row][column]).longValue());
                        } else {
                            System.err.println("Row " + row + ", column " + column +
                                    ": " + objects[row][column].getClass().getSimpleName() + " (" + objects[row][column] +
                                    ") cannot be cast to Long");
                        }
                    }
                }
            } else if (cl == Integer.class) {
                for (int row = 0; row < getRowCount(); row++) {
                    if (isAvailable(row, column)) {
                        if(objects[row][column] instanceof Integer) {
                            out.writeInt(((Integer) objects[row][column]).intValue());
                        } else {
                            System.err.println("Row " + row + ", column " + column +
                                    ": " + objects[row][column].getClass().getSimpleName() + " (" + objects[row][column] +
                                    ") cannot be cast to Integer");
                        }
                    }
                }
            } else if (cl == Short.class) {
                for (int row = 0; row < getRowCount(); row++) {
                    if (isAvailable(row, column)) {
                        if(objects[row][column] instanceof Short) {
                            out.writeShort(((Short) objects[row][column]).shortValue());
                        } else {
                            System.err.println("Row " + row + ", column " + column +
                                    ": " + objects[row][column].getClass().getSimpleName() + " (" + objects[row][column] +
                                    ") cannot be cast to Short");
                        }
                    }
                }
            } else if (cl == Byte.class) {
                for (int row = 0; row < getRowCount(); row++) {
                    if (isAvailable(row, column)) {
                        if(objects[row][column] instanceof Byte) {
                            out.writeByte(((Byte) objects[row][column]).byteValue());
                        } else {
                            System.err.println("Row " + row + ", column " + column +
                                    ": " + objects[row][column].getClass().getSimpleName() + " (" + objects[row][column] +
                                    ") cannot be cast to Byte");
                        }
                    }
                }
            } else if (cl == Boolean.class) {
                for (int row = 0; row < getRowCount(); row++) {
                    if (isAvailable(row, column)) {
                        if(objects[row][column] instanceof Boolean) {
                            out.writeBoolean(((Boolean) objects[row][column]).booleanValue());
                        } else {
                            System.err.println("Row " + row + ", column " + column +
                                    ": " + objects[row][column].getClass().getSimpleName() + " (" + objects[row][column] +
                                    ") cannot be cast to Boolean");
                        }
                    }
                }
            } else if (cl == BufferedImage.class) {
                for (int row = 0; row < getRowCount(); row++) {
                    if (isAvailable(row, column)) {
                        if(objects[row][column] instanceof BufferedImage) {
                            boolean useCache = ImageIO.getUseCache();
                            ImageIO.setUseCache(false);
                            final ByteArrayOutputStream bao = new ByteArrayOutputStream();
                            ImageIO.write((BufferedImage) objects[row][column], "png", bao);
                            final byte[] ba = bao.toByteArray();
                            out.writeObject(ba);
                            ImageIO.setUseCache(useCache);
                        } else {
                            System.err.println("Row " + row + ", column " + column +
                                    ": " + objects[row][column].getClass().getSimpleName() + "(" + objects[row][column] +
                                    ") cannot be cast to BufferedImage");
                        }
                    }
                }
            } else {
                for (int row = 0; row < getRowCount(); row++) {
                    if (isAvailable(row, column)) {
                        out.writeObject(objects[row][column]);
                    }
                }
            }
        }
    }

    @Override
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
        final int version = in.readInt();
        if (version > 0) {
            final int rowCount = in.readInt();
            final int columnCount = in.readInt();
            columnNames = new Object[columnCount];
            classes = new Class[columnCount];
            objects = new Object[rowCount][columnCount];
            for (int column = 0; column < columnCount; column++) {
                columnNames[column] = in.readObject();
            }
            for (int column = 0; column < columnCount; column++) {
                classes[column] = (Class) in.readObject();
            }
            boolean[][] isAvailable = new boolean[rowCount][columnCount];
            for (int row = 0; row < getRowCount(); row++) {
                for (int column = 0; column < getColumnCount(); column++) {
                    isAvailable[row][column] = in.readBoolean();
                }
            }
            for (int column = 0; column < getColumnCount(); column++) {
                final Class cl = getColumnClass(column);
                if (cl == Double.class) {
                    for (int row = 0; row < getRowCount(); row++) {
                        if (isAvailable[row][column]) {
                            Object o = in.readDouble();
                            if (row > 0 && o.equals(objects[row - 1][column])) {
                                o = objects[row - 1][column];
                            }
                            objects[row][column] = o;
                        } else {
                            objects[row][column] = null;
                        }
                    }
                } else if (cl == Float.class) {
                    for (int row = 0; row < getRowCount(); row++) {
                        if (isAvailable[row][column]) {
                            Object o = in.readFloat();
                            if (row > 0 && o.equals(objects[row - 1][column])) {
                                o = objects[row - 1][column];
                            }
                            objects[row][column] = o;
                        } else {
                            objects[row][column] = null;
                        }
                    }
                } else if (cl == Long.class) {
                    for (int row = 0; row < getRowCount(); row++) {
                        if (isAvailable[row][column]) {
                            Object o = in.readLong();
                            if (row > 0 && o.equals(objects[row - 1][column])) {
                                o = objects[row - 1][column];
                            }
                            objects[row][column] = o;
                        } else {
                            objects[row][column] = null;
                        }
                    }
                } else if (cl == Integer.class) {
                    for (int row = 0; row < getRowCount(); row++) {
                        if (isAvailable[row][column]) {
                            Object o = in.readInt();
                            if (row > 0 && o.equals(objects[row - 1][column])) {
                                o = objects[row - 1][column];
                            }
                            objects[row][column] = o;
                        } else {
                            objects[row][column] = null;
                        }
                    }
                } else if (cl == Short.class) {
                    for (int row = 0; row < getRowCount(); row++) {
                        if (isAvailable[row][column]) {
                            Object o = in.readShort();
                            if (row > 0 && o.equals(objects[row - 1][column])) {
                                o = objects[row - 1][column];
                            }
                            objects[row][column] = o;
                        } else {
                            objects[row][column] = null;
                        }
                    }
                } else if (cl == Byte.class) {
                    for (int row = 0; row < getRowCount(); row++) {
                        if (isAvailable[row][column]) {
                            Object o = new Byte(in.readByte());
                            if (row > 0 && o.equals(objects[row - 1][column])) {
                                o = objects[row - 1][column];
                            }
                            objects[row][column] = o;
                        } else {
                            objects[row][column] = null;
                        }
                    }
                } else if (cl == Boolean.class) {
                    for (int row = 0; row < getRowCount(); row++) {
                        if (isAvailable[row][column]) {
                            Object o = new Boolean(in.readBoolean());
                            if (row > 0 && o.equals(objects[row - 1][column])) {
                                o = objects[row - 1][column];
                            }
                            objects[row][column] = o;
                        } else {
                            objects[row][column] = null;
                        }
                    }
                } else if (cl == BufferedImage.class) {
                    for (int row = 0; row < getRowCount(); row++) {
                        if (isAvailable[row][column]) {
                            boolean useCache = ImageIO.getUseCache();
                            ImageIO.setUseCache(false);
                            byte[] array = (byte[]) in.readObject();
                            ByteArrayInputStream bai = new ByteArrayInputStream(array);
                            Object o = ImageIO.read(bai);
                            ImageIO.setUseCache(useCache);
                            if (row > 0 && o.equals(objects[row - 1][column])) {
                                o = objects[row - 1][column];
                            }
                            objects[row][column] = o;
                        } else {
                            objects[row][column] = null;
                        }
                    }
                } else {
                    if (getColumnClass(column) != URL.class) {
                        for (int row = 0; row < getRowCount(); row++) {
                            if (isAvailable[row][column]) {
                                Object o = in.readObject();
                                if (row > 0 && o.equals(objects[row - 1][column])) {
                                    o = objects[row - 1][column];
                                }
                                objects[row][column] = o;
                            } else {
                                objects[row][column] = null;
                            }
                        }
                    } else {
                        // URL.equals() and hashcode() are blocking!
                        for (int row = 0; row < getRowCount(); row++) {
                            if (isAvailable[row][column]) {
                                Object o = in.readObject();
                                objects[row][column] = o;
                            } else {
                                objects[row][column] = null;
                            }
                        }
                    }
                }
            }
        }
    }
}
