package org.molap.dataframe

import com.macrofocus.common.json.*
import org.molap.index.DefaultUniqueIndex
import org.molap.index.IntegerRangeUniqueIndex
import org.molap.index.UniqueIndex
import org.molap.series.Series
import java.io.InputStream
import java.nio.charset.Charset
import java.util.*
import java.util.logging.Logger
import kotlin.reflect.KClass

class JsonDataFrame(private val array: JsonArray) : AbstractDataFrame<Int, String, Any?>() {
    private val labels: Array<String?>
    private val classes: Array<KClass<*>?>

    init {
        val columns = LinkedHashSet<String>()
        val types = HashMap<String, KClass<*>>()
        for (row in 0 until array.length()) {
            val o = array.get<JsonObject>(row)
            for (column in o.keys()) {
                if (!columns.contains(column)) {
                    columns.add(column)
                    val value = o.get<JsonValue>(column)
                    val type = getClass(value)
                    if (type != null) {
                        types[column] = type
                    }
                } else {
                    val value = o.get<JsonValue>(column)
                    val type = getClass(value)
                    if (type != null) {
                        val currentType = types[column]
                        if (currentType == null) {
                            types[column] = type
                        } else if (currentType != type) {
                            types[column] = Any::class
                        }
                    }
                }
            }
        }
        labels = arrayOfNulls(columns.size)
        classes = arrayOfNulls(columns.size)
        var index = 0
        for (column in columns) {
            labels[index] = column
            val type = types[column]
            classes[index] = type ?: Any::class
            index++
        }

//        for (String column : columns) {
//            getLogger().log(Level.INFO, column + ": " + getColumnClass(column));
//        }
    }

    override val rowIndex: UniqueIndex<Int> by lazy { IntegerRangeUniqueIndex(0, array.length() - 1) }
    override val columnIndex: UniqueIndex<String> by lazy {
        val names = ArrayList<String>(labels.size)
        for (c in labels.indices) {
            names.add(labels[c]!!)
        }
        DefaultUniqueIndex<String>(names)
    }

    constructor(json: String) : this((JsonFactory.parse(json) as JsonObject).get<JsonValue>("data") as JsonArray) {}

    override fun getColumnName(column: String): String? {
        return labels[columnIndex.getAddress(column)]
    }

    override fun getColumnClass(column: String): KClass<*> {
        return classes[columnIndex.getAddress(column)]!!
    }

    override val rowCount: Int
        get() = array.length()

    override fun getRowClass(row: Int): KClass<*>? {
        return Any::class
    }

    override fun getValueAt(row: Int, column: String): Any? {
        val jsonObject = array.get<JsonObject>(row)
        val jsonValue = jsonObject.get<JsonValue>(column)

        /*
         * This is subject to the bug where number 0 is treated as null! See
         * https://github.com/gwtproject/gwt/issues/9484
         */return if (jsonValue != null && (jsonValue.type === JsonType.NULL || jsonValue.type == null)) {
            null
        } else {
            if (getColumnClass(column) == Double::class.java) {
                jsonObject.getNumber(column)
            } else if (getColumnClass(column) == Boolean::class.java) {
                jsonObject.getBoolean(column)
            } else {
                try {
                    getObject(jsonValue)
                } catch (e: AssertionError) {
                    // GWT may throw this exception, probably if the column doesn't exist
                    e.printStackTrace()
                    null
                }
            }
        }
    }

    fun getRow(integer: Int?): Series<String, *>? {
        return null
    }

    fun join(series: Series<*,*>?, strings: Array<String?>?): DataFrame<*, *, *>? {
        return null
    }

    private fun getObject(value: JsonValue?): Any? {
        return if (value != null) {
            if (value.type === JsonType.STRING) {
                value.asString()
            } else if (value.type === JsonType.NUMBER) {
                value.asNumber()
            } else if (value.type === JsonType.BOOLEAN) {
                value.asBoolean()
            } else if (value.type === JsonType.ARRAY) {
                val array = value as JsonArray
                val a = arrayOfNulls<Any>(array.length())
                for (i in 0 until array.length()) {
                    a[i] = getObject(array.get<JsonValue>(i))
                }
                a
            } else if (value.type === JsonType.OBJECT) {
                value.toNative()
            } else if (value.type === JsonType.NULL) {
                null
            } else {
                value.toString()
            }
        } else {
            null
        }
    }

    private fun getClass(value: JsonValue): KClass<*>? {
        return if (value.type === JsonType.NULL) {
            null
        } else if (value.type === JsonType.STRING) {
            String::class
        } else if (value.type === JsonType.NUMBER) {
            Double::class
        } else if (value.type === JsonType.BOOLEAN) {
            Boolean::class
        } else if (value.type === JsonType.ARRAY) {
            Array<Any>::class
        } else if (value.type === JsonType.OBJECT) {
            JsonObject::class
        } else {
            String::class
        }
    }

    companion object {
        @JvmStatic
        fun fromInputStream(inputStream : InputStream) : JsonDataFrame {
            val json = String(readAllBytes(inputStream), Charset.forName("UTF-8"))
            return JsonDataFrame(json)
        }

        private val logger: Logger
            private get() = Logger.getLogger(JsonDataFrame::class.java.name)
    }
}

/**
 * Java 9 provides this method
 */
internal fun readAllBytes(inputStream : InputStream, len : Int = Integer.MAX_VALUE) : ByteArray {
    with(inputStream) {
        require(len >= 0) { "len < 0" }
        var bufs: MutableList<ByteArray>? = null
        var result: ByteArray? = null
        var total = 0
        var remaining = len
        var n: Int
        do {
            val defaultBufferSize = 8192
            val buf = ByteArray(Math.min(remaining, defaultBufferSize))
            var nread = 0

            // read to EOF which may read more or less than buffer size
            while (read(
                    buf, nread,
                    Math.min(buf.size - nread, remaining)
                ).also { n = it } > 0
            ) {
                nread += n
                remaining -= n
            }
            if (nread > 0) {
                val maxBufferSize = Integer.MAX_VALUE - 8
                if (maxBufferSize - total < nread) {
                    throw OutOfMemoryError("Required array size too large")
                }
                total += nread
                if (result == null) {
                    result = buf
                } else {
                    if (bufs == null) {
                        bufs = ArrayList()
                        bufs.add(result)
                    }
                    bufs.add(buf)
                }
            }
            // if the last call to read returned -1 or the number of bytes
            // requested have been read then break
        } while (n >= 0 && remaining > 0)
        if (bufs == null) {
            if (result == null) {
                return ByteArray(0)
            }
            return if (result.size == total) result else Arrays.copyOf(result, total)
        }
        result = ByteArray(total)
        var offset = 0
        remaining = total
        for (b in bufs) {
            val count = Math.min(b.size, remaining)
            System.arraycopy(b, 0, result, offset, count)
            offset += count
            remaining -= count
        }
        return result
    }
}