package org.molap.exporter

import org.molap.dataframe.DataFrame
import java.io.IOException
import java.math.BigDecimal
import java.math.MathContext
import java.math.RoundingMode
import java.text.DateFormat
import java.text.DecimalFormat
import java.text.DecimalFormatSymbols
import java.text.SimpleDateFormat
import java.util.*


/**
 * Created by luc on 14/04/16.
 */
abstract class AbstractDataFrameWriter : DataFrameWriter {
    private var timeZone = TimeZone.getTimeZone("UTC")
    private val containsTime: MutableMap<Any?, Boolean> = HashMap()

    @Throws(IOException::class)
    override fun start() {
    }

    fun setTimeZone(timeZone: TimeZone) {
        this.timeZone = timeZone
    }

    fun <Row,Column,Value> dateToString(dataFrame: DataFrame<Row,Column,Value>, column: Column, date: Date?): String {
        return dateToString(dataFrame, column, date, timeZone)
    }

    fun <Row,Column,Value> dateToString(dataFrame: DataFrame<Row,Column,Value>, column: Column, date: Date?, timeZone: TimeZone): String {
        val dateFormat = getDateFormat(dataFrame, column, timeZone)
        return dateFormat.format(date)
    }

    private fun <Row,Column,Value> getDateFormat(dataFrame: DataFrame<Row,Column,Value>, column: Column, timeZone: TimeZone): DateFormat {
        val dateFormat: DateFormat
        dateFormat = if (hasTime(dataFrame, column, timeZone)) {
            SimpleDateFormat("dd.MM.yyyy HH:mm:ss")
        } else {
            SimpleDateFormat("yyyy-MM-dd")
        }
        dateFormat.setTimeZone(timeZone)
        return dateFormat
    }

    protected fun <Row,Column,Value> hasTime(dataFrame: DataFrame<Row,Column,Value>, column: Column, timeZone: TimeZone?): Boolean {
        if (!containsTime.containsKey(column)) {
            var containsTime = false
            for (row in dataFrame.rows()) {
                val value = dataFrame.getValueAt(row, column) as Date?
                if (value != null) {
                    val calendar = Calendar.getInstance(timeZone)
                    calendar.time = value
                    if (calendar[Calendar.HOUR_OF_DAY] != 0 || calendar[Calendar.MINUTE] != 0 || calendar[Calendar.SECOND] != 0) {
                        containsTime = true
                    }
                }
            }
            this.containsTime[column] = containsTime
        }
        return containsTime[column]!!
    }

    companion object {
        private val decimalFormat = DecimalFormat("#################0.#################", DecimalFormatSymbols(Locale.ENGLISH))
        fun numberToString(value: Number?): String {
            if (value == null) {
                return ""
            }
            return if (value is Double) {
                val v = value.toDouble()
                if (java.lang.Double.isNaN(v) || java.lang.Double.isInfinite(v)) {
                    decimalFormat.format(value)
                } else BigDecimal(v, MathContext(15, RoundingMode.HALF_EVEN)).stripTrailingZeros().toPlainString()
            } else if (value is Float) {
                val v = value.toFloat()
                if (java.lang.Float.isNaN(v) || java.lang.Float.isInfinite(v)) {
                    decimalFormat.format(value)
                } else BigDecimal(v.toDouble(), MathContext(6, RoundingMode.HALF_EVEN)).stripTrailingZeros().toPlainString()
            } else if (value is Long) {
                val v = value.toLong()
                java.lang.Long.toString(v)
            } else if (value is Int) {
                val v = value.toInt()
                Integer.toString(v)
            } else if (value is Byte) {
                val v = value.toByte()
                java.lang.Byte.toString(v)
            } else if (value is Short) {
                val v = value.toShort()
                java.lang.Short.toString(v)
            } else if (value is BigDecimal) {
                value.toPlainString()
            } else {
                value.toString()
            }
        }
    }
}
