/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.exporter.parquet

import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.fs.Path
import org.apache.parquet.hadoop.ParquetFileWriter
import org.apache.parquet.hadoop.ParquetWriter
import org.apache.parquet.hadoop.metadata.CompressionCodecName
import org.apache.parquet.hadoop.util.HadoopOutputFile
import org.apache.parquet.io.OutputFile
import org.molap.dataframe.DataFrame
import java.io.File
import java.io.IOException
import java.io.OutputStream

class ParquetDataFrameExporter<R, C, V>(val file: OutputFile,
                               val compressionCodecName: CompressionCodecName = CompressionCodecName.UNCOMPRESSED) {
    var parquetWriter: ParquetWriter<R>? = null

    constructor(file: File) : this(HadoopOutputFile.fromPath(Path(file.toString()), Configuration()))

    constructor(outputStream: OutputStream, compressionCodecName: CompressionCodecName = CompressionCodecName.UNCOMPRESSED) : this(ParquetOutputFile(outputStream), compressionCodecName)

    @Throws(IOException::class)
    fun exportParquet(df: DataFrame<R, C, V>) {
        write(df)
        close()
    }

    @Throws(IOException::class)
    private fun init(df: DataFrame<R, C, V>) : ParquetWriter<R> {
        // set Parquet file block size and page size values
        val pageSize = 64 * 1024

        // create a parquet writer using builder
        val parquetWriter = DataFrameParquetWriter.builder<R,C,V>(file)
            .withSchema(df)
            .withCompressionCodec(compressionCodecName)
            .withPageSize(pageSize)
            .withWriteMode(ParquetFileWriter.Mode.OVERWRITE)
            .build()

        return parquetWriter
    }

    @Throws(IOException::class)
    fun write(df: DataFrame<R, C, V>) {
        if(parquetWriter == null) {
            parquetWriter = init(df)
        }

        for (row in df.rows()) {
            parquetWriter!!.write(row)
        }
    }

    @Throws(IOException::class)
    fun close() {
        parquetWriter!!.close()
    }
}