/*
* Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
*/
@file:OptIn(ExperimentalTime::class)

package org.molap.parquet

import com.macrofocus.common.units.Quantity
import kotlin.time.Instant
import org.apache.hadoop.conf.Configuration
import org.apache.parquet.schema.*
import org.apache.parquet.schema.PrimitiveType.PrimitiveTypeName
import org.apache.parquet.schema.Type.Repetition
import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.geom.LineString
import org.locationtech.jts.geom.Point
import org.locationtech.jts.geom.Polygon
import org.molap.dataframe.DataFrame
import org.molap.datetime.DateTimeTz
import java.net.URL
import kotlin.reflect.KClass
import kotlin.time.ExperimentalTime

/**
 *
 *
 * Converts an DataFrame schema into a Parquet schema, or vice versa. See package
 * documentation for details of the mapping.
 *
 */
class DataFrameSchemaConverter {
    constructor() {}
    constructor(conf: Configuration?) {}

    fun <R, C, V> convert(dataFrame: DataFrame<R, C, V>): MessageType {
        // Typically result in schema starting with
        // required group field_id=-1 AccessorRowMajorDataFrame { ... }
        return MessageType(dataFrame::class.simpleName, convertFields<R, C, V>(dataFrame))
    }

    private fun <R, C, V> convertFields(dataFrame: DataFrame<R, C, V>): List<Type> {
        val types: MutableList<Type> = ArrayList()
        for (field in dataFrame.columns()) {
            types.add(convertField(dataFrame, field))
        }
        return types
    }

    private fun <R, C, V> convertField(fieldName: String, dataFrame: DataFrame<R, C, V>, column: C): Type {
        return convertField(fieldName, dataFrame, column, Repetition.OPTIONAL)
    }

    private fun <R, C, V> convertField(
        fieldName: String,
        dataFrame: DataFrame<R, C, V>,
        column: C,
        repetition: Repetition
    ): Type {
        val builder: Types.PrimitiveBuilder<PrimitiveType>
        val type: KClass<*> = dataFrame.getColumnClass(column)
        builder = if (type == Boolean::class) {
            Types.primitive(PrimitiveTypeName.BOOLEAN, repetition)
        } else if (type == Int::class) {
            // ToDo: Allow for non-null values
//            Types.primitive(PrimitiveTypeName.INT32, if(dataFrame.getColumnClass(column) == Int::class) Repetition.REQUIRED else Repetition.OPTIONAL)
            Types.primitive(PrimitiveTypeName.INT32, repetition)
        } else if (type == Long::class) {
            Types.primitive(PrimitiveTypeName.INT64, repetition)
        } else if (type == Float::class) {
            Types.primitive(PrimitiveTypeName.FLOAT, repetition)
        } else if (type == Double::class) {
            Types.primitive(PrimitiveTypeName.DOUBLE, repetition)
        } else if (type == Quantity::class) {
            Types.primitive(PrimitiveTypeName.DOUBLE, repetition)
        } else if (type == ByteArray::class) {
            Types.primitive(PrimitiveTypeName.BINARY, repetition)
        } else if (type == String::class) {
            Types.primitive(PrimitiveTypeName.BINARY, repetition)
                .`as`(LogicalTypeAnnotation.stringType())
        } else if (type == DateTimeTz::class) {
            Types.primitive(PrimitiveTypeName.INT64, repetition)
                .`as`(LogicalTypeAnnotation.timestampType(true, LogicalTypeAnnotation.TimeUnit.MILLIS))
            //        } else if (type.equals(Object.class)) {
//            return new GroupType(repetition, fieldName, convertFields(schema.getFields()));
        } else if (type == Instant::class) {
            Types.primitive(PrimitiveTypeName.INT64, repetition)
                .`as`(LogicalTypeAnnotation.timestampType(true, LogicalTypeAnnotation.TimeUnit.MILLIS))
        } else if (type == java.time.LocalDateTime::class) {
            Types.primitive(PrimitiveTypeName.INT64, repetition)
                .`as`(LogicalTypeAnnotation.timestampType(true, LogicalTypeAnnotation.TimeUnit.MILLIS))
        } else if (type == java.sql.Timestamp::class) {
            Types.primitive(PrimitiveTypeName.INT64, repetition)
                .`as`(LogicalTypeAnnotation.timestampType(true, LogicalTypeAnnotation.TimeUnit.MILLIS))
        } else if (type == Enum::class) {
            Types.primitive(PrimitiveTypeName.BINARY, repetition)
                .`as`(LogicalTypeAnnotation.enumType())
        } else if (type == URL::class) {
            Types.primitive(PrimitiveTypeName.BINARY, repetition)
                .`as`(LogicalTypeAnnotation.stringType())
        } else if (type == Geometry::class || type == Point::class ||type == LineString::class || type == Polygon::class) {
            Types.primitive(PrimitiveTypeName.BINARY, repetition)
                .`as`(LogicalTypeAnnotation.stringType())
            //        } else if (type.equals(Object[].class)) {
//            if (writeOldListStructure) {
//                return ConversionPatterns.listType(repetition, fieldName,
//                        convertField("array", schema.getElementType(), REPEATED));
//            } else {
//                return ConversionPatterns.listOfElements(repetition, fieldName,
//                        convertField(DataFrameWriteSupport.LIST_ELEMENT_NAME, schema.getElementType()));
//            }
//        } else if (type.equals(Schema.Type.MAP)) {
//            Type valType = convertField("value", schema.getValueType());
////             map key type is always string
//            return ConversionPatterns.stringKeyMapType(repetition, fieldName, valType);
//        } else if (type.equals(Schema.Type.FIXED)) {
//            builder = Types.primitive(FIXED_LEN_BYTE_ARRAY, repetition)
//                    .length(schema.getFixedSize());
//        } else if (type.equals(Schema.Type.UNION)) {
//            return convertUnion(fieldName, schema, repetition);
        } else {
            throw UnsupportedOperationException("Cannot convert type $type for field $fieldName")
        }

        // schema translation can only be done for known logical types because this
        // creates an equivalence
//        LogicalType logicalType = schema.getLogicalType();
//        if (logicalType != null) {
//            if (logicalType instanceof LogicalTypes.Decimal) {
//                LogicalTypes.Decimal decimal = (LogicalTypes.Decimal) logicalType;
//                builder = builder.as(decimalType(decimal.getScale(), decimal.getPrecision()));
//            } else {
//                LogicalTypeAnnotation annotation = convertLogicalType(logicalType);
//                if (annotation != null) {
//                    builder.as(annotation);
//                }
//            }
//        }
        return builder.named(fieldName)
    }

    //    private Type convertUnion(String fieldName, DataFrame schema, Type.Repetition repetition) {
    //        List<Schema> nonNullSchemas = new ArrayList<Schema>(schema.getTypes().size());
    //        // Found any schemas in the union? Required for the edge case, where the union contains only a single type.
    //        boolean foundNullSchema = false;
    //        for (Schema childSchema : schema.getTypes()) {
    //            if (childSchema.getType().equals(Schema.Type.NULL)) {
    //                foundNullSchema = true;
    //                if (Type.Repetition.REQUIRED == repetition) {
    //                    repetition = Type.Repetition.OPTIONAL;
    //                }
    //            } else {
    //                nonNullSchemas.add(childSchema);
    //            }
    //        }
    //        // If we only get a null and one other type then its a simple optional field
    //        // otherwise construct a union container
    //        switch (nonNullSchemas.size()) {
    //            case 0:
    //                throw new UnsupportedOperationException("Cannot convert Avro union of only nulls");
    //
    //            case 1:
    //                return foundNullSchema ? convertField(fieldName, nonNullSchemas.get(0), repetition) :
    //                        convertUnionToGroupType(fieldName, repetition, nonNullSchemas);
    //
    //            default: // complex union type
    //                return convertUnionToGroupType(fieldName, repetition, nonNullSchemas);
    //        }
    //    }
    //    private Type convertUnionToGroupType(String fieldName, Type.Repetition repetition, List<Schema> nonNullSchemas) {
    //        List<Type> unionTypes = new ArrayList<Type>(nonNullSchemas.size());
    //        int index = 0;
    //        for (Schema childSchema : nonNullSchemas) {
    //            unionTypes.add( convertField("member" + index++, childSchema, Type.Repetition.OPTIONAL));
    //        }
    //        return new GroupType(repetition, fieldName, unionTypes);
    //    }
    private fun <R, C, V> convertField(dataFrame: DataFrame<R, C, V>, field: C): Type {
        return convertField<R, C, V>(dataFrame.getColumnName(field)!!, dataFrame, field)
    }

    /**
     * Implements the rules for interpreting existing data from the logical type
     * spec for the LIST annotation. This is used to produce the expected schema.
     *
     *
     * The DataFrameArrayConverter will decide whether the repeated type is the array
     * element type by testing whether the element schema and repeated type are
     * the same. This ensures that the LIST rules are followed when there is no
     * schema and that a schema can be provided to override the default behavior.
     */
    private fun isElementType(repeatedType: Type, parentName: String): Boolean {
        // can't be a synthetic layer because it would be invalid
        return repeatedType.isPrimitive || repeatedType.asGroupType().fieldCount > 1 ||
                repeatedType.asGroupType().getType(0)
                    .isRepetition(Repetition.REPEATED) || repeatedType.name == "array" || repeatedType.name == parentName + "_tuple" // ||
        // default assumption
        //                        assumeRepeatedIsListElement
    }

    companion object {
        private const val ADD_LIST_ELEMENT_RECORDS_DEFAULT = true
    }
}