/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
@file:OptIn(ExperimentalTime::class)

package org.molap.parquet

import com.macrofocus.common.units.Quantity
import kotlin.time.Instant
import org.apache.hadoop.conf.Configuration
import org.apache.parquet.Preconditions
import org.apache.parquet.hadoop.api.WriteSupport
import org.apache.parquet.io.api.Binary
import org.apache.parquet.io.api.RecordConsumer
import org.apache.parquet.schema.GroupType
import org.apache.parquet.schema.MessageType
import org.apache.parquet.schema.Type
import org.apache.parquet.schema.Type.Repetition
import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.geom.LineString
import org.locationtech.jts.geom.Point
import org.locationtech.jts.geom.Polygon
import org.locationtech.jts.io.WKTWriter
import org.molap.dataframe.DataFrame
import org.molap.datetime.DateTimeTz
import java.net.URL
import java.nio.ByteBuffer
import java.time.ZoneOffset
import kotlin.reflect.KClass
import kotlin.time.ExperimentalTime

/**
 * MOLAP implementation of [WriteSupport] for generic, specific, and
 * reflect models. Use [DataFrameParquetWriter] or
 * [ParquetDataFrameExporter] rather than using this class directly.
 */
class DataFrameWriteSupport<R,C,V> : WriteSupport<R> {
    private var recordConsumer: RecordConsumer? = null
    private var rootSchema: MessageType? = null
    private var rootdataFrame: DataFrame<R,C,V>? = null

    //    private Conversion<?> rootConversion;
    //    private DataFrameWriteSupport.ListWriter listWriter;
    constructor() {}

    /**
     * @param schema the write Parquet schema
     * @param dataFrame the write MOLAP schema
     */
    @Deprecated("will be removed in 2.0.0")
    constructor(schema: MessageType?, dataFrame: DataFrame<R,C,V>?) {
        rootSchema = schema
        rootdataFrame = dataFrame
    }

    override fun getName(): String {
        return "avro"
    }

    override fun init(configuration: Configuration): WriteContext {
        if (rootdataFrame == null) {
            rootSchema = DataFrameSchemaConverter().convert<R,C,V>(rootdataFrame!!)
        }

//        boolean writeOldListStructure = configuration.getBoolean(
//                WRITE_OLD_LIST_STRUCTURE, WRITE_OLD_LIST_STRUCTURE_DEFAULT);
//        if (writeOldListStructure) {
//            this.listWriter = new DataFrameWriteSupport.TwoLevelListWriter();
//        } else {
//            this.listWriter = new DataFrameWriteSupport.ThreeLevelListWriter();
//        }
        val extraMetaData: Map<String, String> = HashMap()
        return WriteContext(rootSchema, extraMetaData)
    }

    override fun prepareForWrite(recordConsumer: RecordConsumer) {
        this.recordConsumer = recordConsumer
    }

    override fun write(record: R) {
//        if (rootLogicalType != null) {
//            Conversion<?> conversion = model.getConversionByClass(
//                    record.getClass(), rootLogicalType);
//
//            recordConsumer.startMessage();
//            writeRecordFields(rootSchema, rootdataFrame,
//                    convert(rootdataFrame, rootLogicalType, conversion, record));
//            recordConsumer.endMessage();
//
//        } else {
        recordConsumer!!.startMessage()
        writeRecordFields<R, C, V>(rootSchema, rootdataFrame!!, record)
        recordConsumer!!.endMessage()
        //        }
    }

    private fun writeRecord(
        schema: GroupType, dataFrame: DataFrame<R,C,V>,
        record: R
    ) {
        recordConsumer!!.startGroup()
        writeRecordFields<R, C, V>(schema, dataFrame, record)
        recordConsumer!!.endGroup()
    }

    private fun <R, C, V> writeRecordFields(
        schema: GroupType?, dataFrame: DataFrame<R, C, V>,
        record: R
    ) {
        val fields = schema!!.fields
        var index = 0
        for (column in dataFrame.columns()) {
            val fieldType = fields[index]
            val value: Any? = dataFrame.getValueAt(record, column)
            if (value != null) {
                val columnName = dataFrame.getColumnName(column)
                recordConsumer!!.startField(columnName, index)
                writeValue<R, C, V>(null, dataFrame, column, value)
                recordConsumer!!.endField(columnName, index)
            } else if (fieldType.isRepetition(Repetition.REQUIRED)) {
                throw RuntimeException("Null-value for required field: $column")
            }
            index++
        }
    }
    //    private <V> void writeMap(GroupType schema, Schema dataFrame,
    //                              Map<CharSequence, V> map) {
    //        GroupType innerGroup = schema.getType(0).asGroupType();
    //        Type keyType = innerGroup.getType(0);
    //        Type valueType = innerGroup.getType(1);
    //
    //        recordConsumer.startGroup(); // group wrapper (original type MAP)
    //        if (map.size() > 0) {
    //            recordConsumer.startField(MAP_REPEATED_NAME, 0);
    //
    //            for (Map.Entry<CharSequence, V> entry : map.entrySet()) {
    //                recordConsumer.startGroup(); // repeated group key_value, middle layer
    //                recordConsumer.startField(MAP_KEY_NAME, 0);
    //                writeValue(keyType, MAP_KEY_SCHEMA, entry.getKey());
    //                recordConsumer.endField(MAP_KEY_NAME, 0);
    //                V value = entry.getValue();
    //                if (value != null) {
    //                    recordConsumer.startField(MAP_VALUE_NAME, 1);
    //                    writeValue(valueType, dataFrame.getValueType(), value);
    //                    recordConsumer.endField(MAP_VALUE_NAME, 1);
    //                } else if (!valueType.isRepetition(Type.Repetition.OPTIONAL)) {
    //                    throw new RuntimeException("Null map value for " + dataFrame.getName());
    //                }
    //                recordConsumer.endGroup();
    //            }
    //
    //            recordConsumer.endField(MAP_REPEATED_NAME, 0);
    //        }
    //        recordConsumer.endGroup();
    //    }
    //    private void writeUnion(GroupType parquetSchema, Schema dataFrame,
    //                            Object value) {
    //        recordConsumer.startGroup();
    //
    //        // ResolveUnion will tell us which of the union member types to
    //        // deserialise.
    //        int avroIndex = model.resolveUnion(dataFrame, value);
    //
    //        // For parquet's schema we skip nulls
    //        GroupType parquetGroup = parquetSchema.asGroupType();
    //        int parquetIndex = avroIndex;
    //        for (int i = 0; i < avroIndex; i++) {
    //            if (dataFrame.getTypes().get(i).getType().equals(Schema.Type.NULL)) {
    //                parquetIndex--;
    //            }
    //        }
    //
    //        // TODO: what if the value is null?
    //
    //        // Sparsely populated method of encoding unions, each member has its own
    //        // set of columns.
    //        String memberName = "member" + parquetIndex;
    //        recordConsumer.startField(memberName, parquetIndex);
    //        writeValue(parquetGroup.getType(parquetIndex),
    //                dataFrame.getTypes().get(avroIndex), value);
    //        recordConsumer.endField(memberName, parquetIndex);
    //
    //        recordConsumer.endGroup();
    //    }
    /**
     * Calls an appropriate write method based on the value.
     * Value MUST not be null.
     *
     * @param type the Parquet type
     * @param dataFrame the MOLAP schema
     * @param value a non-null value to write
     */
    private fun <R, C, V> writeValue(type: Type?, dataFrame: DataFrame<R,C,V>, column: C, value: Any) {
        writeValueWithoutConversion<R, C, V>(type, dataFrame, column, value)
    }

    /**
     * Calls an appropriate write method based on the value.
     * Value must not be null and the schema must not be nullable.
     *
     * @param type a Parquet type
     * @param dataFrame a non-nullable Avro schema
     * @param value a non-null value to write
     */
    private fun <R, C, V> writeValueWithoutConversion(
        type: Type?,
        dataFrame: DataFrame<R, C, V>,
        column: C,
        value: Any
    ) {
        val columnClass: KClass<*> = dataFrame.getColumnClass(column)
        when {
            Boolean::class == columnClass -> {
                recordConsumer!!.addBoolean((value as Boolean))
            }
            Int::class == columnClass -> {
                recordConsumer!!.addInteger((value as Number).toInt())
            }
            value is Char -> {
                recordConsumer!!.addInteger((value as Char).code)
            }
            Long::class == columnClass -> {
                recordConsumer!!.addLong((value as Number).toLong())
            }
            Float::class == columnClass -> {
                recordConsumer!!.addFloat((value as Number).toFloat())
            }
            Double::class == columnClass -> {
                if(!(value is Number)) {
                    println("Column $column has value $value of type ${value::class} instead of Number")
                    throw ClassCastException("Column $column has value $value of type ${value::class} instead of Number")
                }
                recordConsumer!!.addDouble((value as Number).toDouble())
                //            case FIXED:
                //                recordConsumer.addBinary(Binary.fromReusedByteArray(((GenericFixed) value).bytes()));
                //                break;
            }
            Quantity::class == columnClass -> {
                recordConsumer!!.addDouble((value as Quantity<*>).amount)
                //            case FIXED:
                //                recordConsumer.addBinary(Binary.fromReusedByteArray(((GenericFixed) value).bytes()));
                //                break;
            }
            ByteArray::class == columnClass -> {
                if (value is ByteArray) {
                    recordConsumer!!.addBinary(Binary.fromReusedByteArray(value))
                } else {
                    recordConsumer!!.addBinary(Binary.fromReusedByteBuffer(value as ByteBuffer))
                }
            }
            DateTimeTz::class == columnClass -> {
                recordConsumer!!.addLong((value as DateTimeTz).datetime.toEpochMilliseconds())
            }
            Instant::class == columnClass -> {
                recordConsumer!!.addLong((value as Instant).toEpochMilliseconds())
            }
            java.time.LocalDateTime::class == columnClass -> {
                recordConsumer!!.addLong((value as java.time.LocalDateTime).toEpochSecond(ZoneOffset.UTC) * 1000)
            }
            java.sql.Timestamp::class == columnClass -> {
                recordConsumer!!.addLong((value as java.sql.Timestamp).time)
            }
            String::class == columnClass -> {
                recordConsumer!!.addBinary(fromString(value))
                //            case RECORD:
    //                writeRecord(type.asGroupType(), dataFrame, value);
    //                break;
            }
            Enum::class == columnClass -> {
                recordConsumer!!.addBinary(Binary.fromString(value.toString()))
            }
            URL::class == columnClass -> {
                recordConsumer!!.addBinary(fromURL(value))
                //            case ARRAY:
    //                listWriter.writeList(type.asGroupType(), dataFrame, value);
    //                break;
    //            case MAP:
    //                writeMap(type.asGroupType(), dataFrame, (Map<CharSequence, ?>) value);
    //                break;
    //            case UNION:
    //                writeUnion(type.asGroupType(), dataFrame, value);
    //                break;
            }
            Geometry::class == columnClass -> {
                recordConsumer!!.addBinary(fromGeometry(value))
            }
            Point::class == columnClass -> {
                recordConsumer!!.addBinary(fromGeometry(value))
            }
            LineString::class == columnClass -> {
                recordConsumer!!.addBinary(fromGeometry(value))
            }
            Polygon::class == columnClass -> {
                recordConsumer!!.addBinary(fromGeometry(value))
            }
            else -> { println("Unknown column class $columnClass for $column and value $value") }
        }
    }

    private fun fromString(value: Any): Binary {
        return if (value is CharSequence) {
            Binary.fromCharSequence(value)
        } else Binary.fromCharSequence(value.toString())
    }

    private fun fromURL(value: Any): Binary? {
        return if (value is URL) {
            Binary.fromCharSequence(value.toExternalForm())
        } else null
    }

    private fun fromGeometry(value: Any): Binary? {
        return if (value is Geometry) {
            Binary.fromCharSequence(WKTWriter().write(value))
        } else null
    }

    private abstract inner class ListWriter {
        protected abstract fun writeCollection(
            type: GroupType?, schema: DataFrame<R,C,V>, collection: Collection<*>?
        )

        protected abstract fun writeObjectArray(
            type: GroupType?, schema: DataFrame<R,C,V>, array: Array<Any?>?
        )

        protected abstract fun startArray()
        protected abstract fun endArray()
        fun writeList(schema: GroupType?, dataFrame: DataFrame<R,C,V>, value: Any) {
            recordConsumer!!.startGroup() // group wrapper (original type LIST)
            if (value is Collection<*>) {
                writeCollection(schema, dataFrame, value)
            } else {
                val arrayClass: Class<*> = value.javaClass
                Preconditions.checkArgument(
                    arrayClass.isArray,
                    "Cannot write unless collection or array: " + arrayClass.name
                )
                writeJavaArray(schema, dataFrame, arrayClass, value)
            }
            recordConsumer!!.endGroup()
        }

        fun writeJavaArray(
            schema: GroupType?, dataFrame: DataFrame<R,C,V>,
            arrayClass: Class<*>, value: Any
        ) {
            val elementClass = arrayClass.componentType
            if (!elementClass.isPrimitive) {
                writeObjectArray(schema, dataFrame, value as Array<Any?>)
                return
            }
            val columnClass: KClass<*> = dataFrame.getColumnClass(null as C)
            if (Boolean::class.java == columnClass) {
                Preconditions.checkArgument(
                    elementClass == Boolean::class.javaPrimitiveType,
                    "Cannot write as boolean array: " + arrayClass.name
                )
                writeBooleanArray(value as BooleanArray)
            } else if (Int::class.java == columnClass) {
                if (elementClass == Byte::class.javaPrimitiveType) {
                    writeByteArray(value as ByteArray)
                } else if (elementClass == Char::class.javaPrimitiveType) {
                    writeCharArray(value as CharArray)
                } else if (elementClass == Short::class.javaPrimitiveType) {
                    writeShortArray(value as ShortArray)
                } else if (elementClass == Int::class.javaPrimitiveType) {
                    writeIntArray(value as IntArray)
                } else {
                    throw IllegalArgumentException(
                        "Cannot write as an int array: " + arrayClass.name
                    )
                }
            } else if (Long::class.java == columnClass) {
                Preconditions.checkArgument(
                    elementClass == Long::class.javaPrimitiveType,
                    "Cannot write as long array: " + arrayClass.name
                )
                writeLongArray(value as LongArray)
            } else if (Float::class.java == columnClass) {
                Preconditions.checkArgument(
                    elementClass == Float::class.javaPrimitiveType,
                    "Cannot write as float array: " + arrayClass.name
                )
                writeFloatArray(value as FloatArray)
            } else if (Double::class.java == columnClass) {
                Preconditions.checkArgument(
                    elementClass == Double::class.javaPrimitiveType,
                    "Cannot write as double array: " + arrayClass.name
                )
                writeDoubleArray(value as DoubleArray)
            } else {
                throw IllegalArgumentException(
                    "Cannot write " +
                            dataFrame.getColumnClass(null as C).toString() + " array: " + arrayClass.name
                )
            }
        }

        protected fun writeBooleanArray(array: BooleanArray) {
            if (array.size > 0) {
                startArray()
                for (element in array) {
                    recordConsumer!!.addBoolean(element)
                }
                endArray()
            }
        }

        protected fun writeByteArray(array: ByteArray) {
            if (array.size > 0) {
                startArray()
                for (element in array) {
                    recordConsumer!!.addInteger(element.toInt())
                }
                endArray()
            }
        }

        protected fun writeShortArray(array: ShortArray) {
            if (array.size > 0) {
                startArray()
                for (element in array) {
                    recordConsumer!!.addInteger(element.toInt())
                }
                endArray()
            }
        }

        protected fun writeCharArray(array: CharArray) {
            if (array.size > 0) {
                startArray()
                for (element in array) {
                    recordConsumer!!.addInteger(element.code)
                }
                endArray()
            }
        }

        protected fun writeIntArray(array: IntArray) {
            if (array.size > 0) {
                startArray()
                for (element in array) {
                    recordConsumer!!.addInteger(element)
                }
                endArray()
            }
        }

        protected fun writeLongArray(array: LongArray) {
            if (array.size > 0) {
                startArray()
                for (element in array) {
                    recordConsumer!!.addLong(element)
                }
                endArray()
            }
        }

        protected fun writeFloatArray(array: FloatArray) {
            if (array.size > 0) {
                startArray()
                for (element in array) {
                    recordConsumer!!.addFloat(element)
                }
                endArray()
            }
        }

        protected fun writeDoubleArray(array: DoubleArray) {
            if (array.size > 0) {
                startArray()
                for (element in array) {
                    recordConsumer!!.addDouble(element)
                }
                endArray()
            }
        }
    } //    private class ThreeLevelListWriter extends DataFrameWriteSupport.ListWriter {

    //        @Override
    //        protected void writeCollection(GroupType type, Schema schema, Collection<?> collection) {
    //            if (collection.size() > 0) {
    //                recordConsumer.startField(LIST_REPEATED_NAME, 0);
    //                GroupType repeatedType = type.getType(0).asGroupType();
    //                Type elementType = repeatedType.getType(0);
    //                for (Object element : collection) {
    //                    recordConsumer.startGroup(); // repeated group array, middle layer
    //                    if (element != null) {
    //                        recordConsumer.startField(LIST_ELEMENT_NAME, 0);
    //                        writeValue(elementType, schema.getElementType(), element);
    //                        recordConsumer.endField(LIST_ELEMENT_NAME, 0);
    //                    } else if (!elementType.isRepetition(Type.Repetition.OPTIONAL)) {
    //                        throw new RuntimeException(
    //                                "Null list element for " + schema.getName());
    //                    }
    //                    recordConsumer.endGroup();
    //                }
    //                recordConsumer.endField(LIST_REPEATED_NAME, 0);
    //            }
    //        }
    //
    //        @Override
    //        protected void writeObjectArray(GroupType type, Schema schema,
    //                                        Object[] array) {
    //            if (array.length > 0) {
    //                recordConsumer.startField(LIST_REPEATED_NAME, 0);
    //                GroupType repeatedType = type.getType(0).asGroupType();
    //                Type elementType = repeatedType.getType(0);
    //                for (Object element : array) {
    //                    recordConsumer.startGroup(); // repeated group array, middle layer
    //                    if (element != null) {
    //                        recordConsumer.startField(LIST_ELEMENT_NAME, 0);
    //                        writeValue(elementType, schema.getElementType(), element);
    //                        recordConsumer.endField(LIST_ELEMENT_NAME, 0);
    //                    } else if (!elementType.isRepetition(Type.Repetition.OPTIONAL)) {
    //                        throw new RuntimeException(
    //                                "Null list element for " + schema.getName());
    //                    }
    //                    recordConsumer.endGroup();
    //                }
    //                recordConsumer.endField(LIST_REPEATED_NAME, 0);
    //            }
    //        }
    //
    //        @Override
    //        protected void startArray() {
    //            recordConsumer.startField(LIST_REPEATED_NAME, 0);
    //            recordConsumer.startGroup(); // repeated group array, middle layer
    //            recordConsumer.startField(LIST_ELEMENT_NAME, 0);
    //        }
    //
    //        @Override
    //        protected void endArray() {
    //            recordConsumer.endField(LIST_ELEMENT_NAME, 0);
    //            recordConsumer.endGroup();
    //            recordConsumer.endField(LIST_REPEATED_NAME, 0);
    //        }
    //    }
    companion object {
        private const val MAP_REPEATED_NAME = "key_value"
        private const val MAP_KEY_NAME = "key"
        private const val MAP_VALUE_NAME = "value"
        private const val LIST_REPEATED_NAME = "list"
        private const val OLD_LIST_REPEATED_NAME = "array"
        const val LIST_ELEMENT_NAME = "element"
    }
}