package org.molap.postgresql

import com.macrofocus.common.units.Quantity
import kotlinx.datetime.Instant
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toKotlinInstant
import kotlinx.datetime.toLocalDateTime
import org.jetbrains.exposed.sql.*
import org.jetbrains.exposed.sql.SchemaUtils.createMissingTablesAndColumns
import org.jetbrains.exposed.sql.SqlExpressionBuilder.between
import org.jetbrains.exposed.sql.SqlExpressionBuilder.eq
import org.jetbrains.exposed.sql.SqlExpressionBuilder.greaterEqSubQuery
import org.jetbrains.exposed.sql.kotlin.datetime.datetime
import org.jetbrains.exposed.sql.kotlin.datetime.timestamp
import org.jetbrains.exposed.sql.transactions.transaction
import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.geom.LineString
import org.locationtech.jts.geom.Point
import org.locationtech.jts.geom.Polygon
import org.molap.dataframe.DataFrame
import org.molap.datetime.DateTimeTz
import org.molap.postgresql.exposed.geometry
import java.time.ZoneOffset

fun <R,C,V> DataFrame<R, C, V>.writePostgreSQL(database : Database, tableName: String) {
    try {
        val table = recreateTable(database, tableName, this)
        insertIntoDynamicTable(database, table, this)
    } catch (e: Exception) {
        e.printStackTrace()
    }
}

fun <R,C,V> DataFrame<R, C, V>.appendPostgreSQL(database : Database, tableName: String) {
    val table = getTable(database, tableName, this)
    insertIntoDynamicTable(database, table, this)
}

fun connectToDatabase(): Database {
    return Database.connect(
//        url = "jdbc:postgresql://localhost:5432/efluxdb",
//        driver = "org.postgresql.Driver",
        url = "jdbc:postgresql_postGIS://localhost:5434/efluxdb",
        driver = "net.postgis.jdbc.DriverWrapperLW",
        user = "postgres",
        password = "zMLjmwMAlx4M056"
    )
}

fun <R, C, V> createTableInDatabase(database: Database, tableName: String, dataFrame: DataFrame<R, C, V>): Table {
    val dynamicTable = createDynamicTable(tableName, dataFrame)

    transaction(database) {
        SchemaUtils.createMissingTablesAndColumns(dynamicTable)
    }

    return dynamicTable
}
fun <R, C, V> createDynamicTable(tableName: String, dataFrame: DataFrame<R, C, V>): Table {
    return object : Table(tableName) {
        init {
            for (column in dataFrame.columns()) {
                val columnType = dataFrame.getColumnClass(column)
                val columnName = dataFrame.getColumnName(column)!!
                when (columnType) {
                    String::class -> {
                        varchar(columnName, length = 255).nullable().index()
                    }
                    Int::class -> integer(columnName).nullable()
                    Boolean::class -> bool(columnName).nullable()
                    Float::class -> float(columnName).nullable()
                    Double::class -> double(columnName).nullable()
                    Quantity::class -> double(columnName).nullable()
                    DateTimeTz::class -> datetime(columnName).nullable().index()
                    Instant::class -> timestamp(columnName).nullable().index()
                    java.time.LocalDateTime::class -> timestamp(columnName).nullable().index()
                    ByteArray::class -> binary(columnName).nullable()
                    Geometry::class -> {
                        val srid = getTrailingFourDigitsOrDefault(columnName)
                        val column = geometry(columnName, srid = srid).nullable()
                        this.index(null, false, column, indexType = "GIST")
                    }
                    Point::class -> {
                        val srid = getTrailingFourDigitsOrDefault(columnName)
                        val column = geometry(columnName, srid = srid).nullable()
                        this.index(null, false, column, indexType = "GIST")
                    }
                    LineString::class -> {
                        val srid = getTrailingFourDigitsOrDefault(columnName)
                        val column = geometry(columnName, srid = srid).nullable()
                        this.index(null, false, column, indexType = "GIST")
                    }
                    Polygon::class -> {
                        val srid = getTrailingFourDigitsOrDefault(columnName)
                        val column = geometry(columnName, srid = srid).nullable()
                        this.index(null, false, column, indexType = "GIST")
                    }
                    else -> throw IllegalArgumentException("Unsupported column type: $columnType for column $columnName of table $tableName")
                }
            }
        }
    }
}

fun getTrailingFourDigitsOrDefault(str: String, default: Int = 4326): Int {
    return Regex("""(\d{4})$""").find(str)?.value?.toIntOrNull() ?: default
}

fun <R, C, V>  insertIntoDynamicTable(database: Database, dynamicTable: Table, dataFrame: DataFrame<R, C, V>) {
    transaction(database) {
        for(row in dataFrame.rows()) {
            dynamicTable.insert {
                for (column in dataFrame.columns()) {
                    var value = dataFrame.getValueAt(row, column)
                    if(value != null) {
                        val columnName = dataFrame.getColumnName(column)!!
                        val dynamicColumn = dynamicTable.columns.find { it.name == columnName }!! as Column<V>
                        if(value is Quantity<*>) {
                            value = value.amount as V
                        } else if(value is DateTimeTz) {
                            value = value.datetime.toLocalDateTime(TimeZone.UTC) as V
                        } else if(value is java.time.LocalDateTime) {
                            value = value.toInstant(ZoneOffset.UTC).toKotlinInstant() as V
    //                    } else if(value is Geometry) {
    //                        value = value.toText() as V
                        }
                        it[dynamicColumn] = value
                    }
                }
            }
        }
    }
}

fun <R, C, V>  replaceIntoDynamicTable(database: Database, dynamicTable: Table, dataFrame: DataFrame<R, C, V>, column: String, from: DateTimeTz, to: DateTimeTz) {
    transaction(database) {
        for(row in dataFrame.rows()) {
            val column = dynamicTable.columns.find { it.name == "column" }!! as Column<Instant>
            dynamicTable.deleteWhere {
                // ToDo: take care of incluse/exclusive range
                column.between(from.datetime, to.datetime)
            }
            dynamicTable.insert {
                for (column in dataFrame.columns()) {
                    var value = dataFrame.getValueAt(row, column)
                    if(value != null) {
                        val columnName = dataFrame.getColumnName(column)!!
                        val dynamicColumn = dynamicTable.columns.find { it.name == columnName }!! as Column<V>
                        if(value is Quantity<*>) {
                            value = value.amount as V
                        } else if(value is DateTimeTz) {
                            value = value.datetime.toLocalDateTime(TimeZone.UTC) as V
                        } else if(value is java.time.LocalDateTime) {
                            value = value.toInstant(ZoneOffset.UTC).toKotlinInstant() as V
                            //                    } else if(value is Geometry) {
                            //                        value = value.toText() as V
                        }
                        it[dynamicColumn] = value
                    }
                }
            }
        }
    }
}

private val map = HashMap<String, Table>()

private fun <R, C, V> recreateTable(database: Database, tableName: String, dataFrame: DataFrame<R, C, V>):
        Table {
    return map.getOrPut(tableName, {
        val dynamicTable = createDynamicTable(tableName, dataFrame)

        transaction(database) {
            SchemaUtils.drop(dynamicTable)
        }

        transaction(database) {
            SchemaUtils.create(dynamicTable)
//            SchemaUtils.createMissingTablesAndColumns(dynamicTable)
        }

        transaction(database) {
            dynamicTable.deleteAll()
        }

        dynamicTable
    })
}

private fun <R, C, V> getTable(database: Database, tableName: String, dataFrame: DataFrame<R, C, V>):
        Table {
    return map.getOrPut(tableName, {
        val dynamicTable = createDynamicTable(tableName, dataFrame)

        transaction(database) {
            SchemaUtils.createMissingTablesAndColumns(dynamicTable)
        }

        dynamicTable
    })
}