package org.molap.postgresql.exposed

import org.jetbrains.exposed.sql.Column
import org.jetbrains.exposed.sql.ColumnType
import org.jetbrains.exposed.sql.Table
import org.jetbrains.exposed.sql.statements.api.PreparedStatementApi
import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.io.WKTReader
import org.locationtech.jts.io.WKTWriter
import net.postgis.jdbc.PGgeometry

class GeometryColumnType(val srid: Int = 4326, val type: String) : ColumnType<Geometry>() {
    private val wktReader = WKTReader() // Parses WKT strings into Geometry objects
    private val wktWriter = WKTWriter() // Converts Geometry objects to WKT strings

    override fun sqlType(): String = "GEOMETRY($type, $srid)"

    override fun valueFromDB(value: Any): Geometry? {
        return when (value) {
            is String -> wktReader.read(value) // Convert WKT string to Geometry
            else -> throw IllegalArgumentException("Unexpected value: $value")
        }
    }

    override fun valueToDB(value: Geometry?): Any? {
        return when (value) {
            null -> null
            is Geometry -> value
            else -> throw IllegalArgumentException("Unexpected value type: $value")
        }
    }

    override fun notNullValueToDB(value: Geometry): Any {
        return valueToDB(value) ?: throw IllegalArgumentException("Value cannot be null")
    }

    override fun setParameter(stmt: PreparedStatementApi, index: Int, value: Any?) {
        if (value == null) {
            stmt.setNull(index, GeometryColumnType(type = "Geometry")) // PostGIS uses custom type `GEOMETRY`
        } else {
            if(value is Geometry) {
                stmt[index] = PGgeometry(value.toText())
            } else {
                stmt[index] = value
            }
        }
    }
}

fun Table.geometry(name: String, type: String = "Geometry", srid: Int = 4326): Column<Geometry> =
    registerColumn(name, GeometryColumnType(srid, type))