package org.molap.questdb

import com.macrofocus.common.json.*
import com.soywiz.klock.DateFormat
import com.soywiz.klock.DateTime
import com.soywiz.klock.parseUtc
import org.molap.dataframe.AbstractDataFrame
import org.molap.index.DefaultUniqueIndex
import org.molap.index.IntegerRangeUniqueIndex
import org.molap.index.UniqueIndex
import kotlin.reflect.KClass

enum class DateTimeHandling {
    String,
    Long,
    DateTime
}

class QuestDBDataFrame(val root: JsonObject, val dateTimeHanding: DateTimeHandling = DateTimeHandling.String): AbstractDataFrame<Int, String, Any?>() {
    private val labels: Array<String?>
    private val classes: Array<KClass<out Any>?>
    private val array: JsonArray?

    constructor(json: String, dateTimeHanding: DateTimeHandling = DateTimeHandling.String) : this(JsonFactory.parse<JsonObject>(json), dateTimeHanding) {
    }

    init {
        val columns = LinkedHashSet<String>()
        val types: HashMap<String, KClass<*>> = HashMap<String, KClass<*>>()

        val jsonColumns = root.getArray("columns")

        if(jsonColumns != null) {
            for (i in 0 until jsonColumns.length()) {
                val column = jsonColumns.getObject(i)
                val name = column?.getString("name")!!
                val type = column?.getString("type")!!.lowercase()

                columns.add(name)
                types[name] = when {
                    type == "boolean" -> Boolean::class
                    type == "byte" -> Number::class
                    type == "short" -> Number::class
                    type == "char" -> String::class
                    type == "int" -> Number::class
                    type == "float" -> Number::class
                    type == "symbol" -> String::class
                    type == "string" -> String::class
                    type == "long" -> Number::class
                    type == "date" -> when(dateTimeHanding) {
                        DateTimeHandling.String -> String::class
                        DateTimeHandling.Long -> Long::class
                        DateTimeHandling.DateTime -> DateTime::class
                    }
                    type == "timestamp" ->when(dateTimeHanding) {
                        DateTimeHandling.String -> String::class
                        DateTimeHandling.Long -> Long::class
                        DateTimeHandling.DateTime -> DateTime::class
                    }
                    type == "double" -> Double::class
                    type == "binary" -> String::class
                    type == "long256" -> Number::class
                    else -> Any::class
                }
            }
        }

        array = root.getArray("dataset")

        labels = arrayOfNulls(columns.size)
        classes = arrayOfNulls<KClass<out Any>>(columns.size)
        var index = 0
        for (column in columns) {
            labels[index] = column
            val type: KClass<*>? = types[column]
            classes[index] = if (type != null) type else Any::class
            index++
        }
    }

    override fun getRowClass(row: Int): KClass<*> {
        return Any::class
    }

    override fun getColumnClass(column: String): KClass<out Any> {
        return classes[columnIndex.getAddress(column)]!!
    }

    override fun getValueAt(row: Int, column: String): Any? {
        return getValueAt(getRowAddress(row), getColumnAddress(column))
    }

    override val rowIndex: UniqueIndex<Int> by lazy { IntegerRangeUniqueIndex(0, (array?.length() ?: 0) - 1) }
    override val columnIndex: UniqueIndex<String> by lazy {
        val names = ArrayList<String>(labels.size)
        for (c in labels.indices) {
            names.add(labels[c]!!)
        }
        DefaultUniqueIndex<String>(names)
    }

    fun getValueAt(rowIndex: Int, columnIndex: Int): Any? {
        return try {
            val value: JsonValue? = array?.getArray(rowIndex)?.get(columnIndex)
            if(value != null) {
                if (value.type === JsonType.NULL) {
                    null
                } else if (value.type == JsonType.ARRAY) {
                    val array: JsonArray = value.asJsonArray()!!
                    val a = arrayOfNulls<Any>(array.length())
                    for (i in 0 until array.length()) {
                        a[i] = array.get(i)
                    }
                    a
                } else if (value.type == JsonType.BOOLEAN) {
                    value.asBoolean()
                } else if (value.type == JsonType.NUMBER) {
                    value.asNumber()
                } else if (value.type == JsonType.STRING) {
                    if(classes[columnIndex] == DateTime::class) {
                        value.asString()?.let {
                            val dateFormat: DateFormat = DateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSz") // Construct a new DateFormat from a String
                            dateFormat.parseUtc(it)
                        }
                    } else if(classes[columnIndex] == Long::class) {
                        value.asString()?.let {
                            val dateFormat: DateFormat = DateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSz") // Construct a new DateFormat from a String
                            dateFormat.parseUtc(it)?.unixMillisLong
                        }
                    } else {
                        value.asString()
                    }
                } else {
                    value
                }
            } else {
                return null
            }
        } catch (e: JsonException) {
            e.printStackTrace()
            null
        }
    }
}