package org.molap.questdb

import com.macrofocus.common.units.Quantity
import com.soywiz.klock.DateTime
import io.questdb.cairo.CairoEngine
import io.questdb.griffin.SqlCompiler
import io.questdb.griffin.SqlExecutionContextImpl
import io.questdb.std.Os
import org.molap.dataframe.DataFrame
import org.molap.exporter.DataFrameExport

class QuestDBDataFrameExport(private val engine: CairoEngine, val table: String, private val initialize : Boolean = true) : DataFrameExport {
    private var initialized = !initialize
    val ctx = SqlExecutionContextImpl(engine, 1)
    val compiler = SqlCompiler(engine)

    private fun <R,C,V>  initialize(dataFrame: DataFrame<R,C,V> ) {
        val sb = StringBuilder("create table $table (")
        var first = true
        var timestamp : String? = null
        for (column in dataFrame.columns()) {
            if(!first) {
                sb.append(", ")
            } else {
                first = false
            }
            sb.append(dataFrame.getColumnName(column))
            sb.append(" ")
            when {
                dataFrame.getColumnClass(column) == Double::class -> {
                    sb.append("double")
                }
                dataFrame.getColumnClass(column) == Int::class -> {
                    sb.append("int")
                }
                dataFrame.getColumnClass(column) == Boolean::class -> {
                    sb.append("boolean")
                }
                dataFrame.getColumnClass(column) == DateTime::class -> {
                    sb.append("timestamp")
                    if(timestamp == null) {
                        timestamp = dataFrame.getColumnName(column)
                    }
                }
                dataFrame.getColumnClass(column) == String::class -> {
        //                sb.append("string")
                    sb.append("symbol")
                }
                else -> {
        //                sb.append("string")
                    sb.append("symbol")
                }
            }
        }
        sb.append(")")
        if(timestamp != null) {
            sb.append("timestamp($timestamp) PARTITION BY MONTH")
        }

        compiler.compile(
            sb.toString(),
            ctx
        )
    }

    override fun <R, C, V> write(dataFrame: DataFrame<R, C, V>) {
        if(!initialized) {
            initialize(dataFrame)

            initialized = true
        }

        var timestamp : C? = null
        for (column in dataFrame.columns()) {
            if(dataFrame.getColumnClass(column) == DateTime::class) {
                if(timestamp == null) {
                    timestamp = column
                }
            }
        }
        engine.getWriter(ctx.cairoSecurityContext, table, "test").use { writer ->
            for (r in dataFrame.rows()) {
                val row = writer.newRow(if(timestamp != null) {(dataFrame.getValueAt(r, timestamp) as DateTime).unixMillisLong * 1000 } else {Os.currentTimeMicros()})

                for (column in dataFrame.columns()) {
                    val c = dataFrame.getColumnAddress(column)
                    val v = dataFrame.getValueAt(r, column)

                    when {
                        dataFrame.getColumnClass(column) == Double::class -> {
                            if(v is Double) {
                                row.putDouble(c, v)
                            } else if(v is Quantity<*>) {
                                row.putDouble(c, v.amount)
                            }
                        }
                        dataFrame.getColumnClass(column) == Int::class -> {
                            if(v is Int) {
                                row.putInt(c, v)
                            }
                        }
                        dataFrame.getColumnClass(column) == Boolean::class -> {
                            if(v is Boolean) {
                                row.putBool(c, v)
                            }
                        }
                        dataFrame.getColumnClass(column) == DateTime::class -> {
                            // If this is the designated timestamp column, then the value is automatically populated by the call to newRow()
                            if (column != timestamp) {
                                if (v is DateTime) {
                                    row.putDate(c, v.unixMillisLong * 1000)
                                }
                            }
                        }
                        dataFrame.getColumnClass(column) == String::class -> {
                            if(v is String) {
                //                            row.putStr(c, v)
                                row.putSym(c, v)
                            } else {
                                if(v != null) {
                                    //                            row.putStr(c, v.toString())
                                    row.putSym(c, v.toString())
                                }
                            }
                        }
                        else -> {
                            if(v != null) {
                //                            row.putStr(c, v.toString())
                                row.putSym(c, v.toString())
                            }
                        }
                    }
                }
                row.append()
            }

            writer.commit()
        }
    }

    override fun close() {
    }
}