package org.molap.questdb

import com.macrofocus.common.units.Quantity
import io.questdb.cairo.CairoEngine
import io.questdb.cairo.security.AllowAllSecurityContext
import io.questdb.griffin.SqlCompilerImpl
import io.questdb.griffin.SqlExecutionContextImpl
import io.questdb.std.Os
import korlibs.time.DateTime
import org.molap.dataframe.DataFrame
import org.molap.exporter.DataFrameExport

class QuestDBDataFrameExport(private val engine: CairoEngine, val table: String, private val lock: String = "lock", private val initialize : Boolean = true) : DataFrameExport {
    private var initialized = !initialize
    val ctx = SqlExecutionContextImpl(engine, 1)
    val compiler = SqlCompilerImpl(engine)

    init {
        ctx.with(AllowAllSecurityContext.INSTANCE, null, null)
    }

    private fun <R,C,V>  initialize(dataFrame: DataFrame<R,C,V> ) {
        val sb = StringBuilder("create table \"$table\" (")
        var first = true
        var timestamp : String? = null
        for (column in dataFrame.columns()) {
            if(!first) {
                sb.append(", ")
            } else {
                first = false
            }
            sb.append('"' + dataFrame.getColumnName(column)!! + '"')
            sb.append(" ")
            when {
                dataFrame.getColumnClass(column) == Double::class -> {
                    sb.append("double")
                }
                dataFrame.getColumnClass(column) == Float::class -> {
                    sb.append("float")
                }
                dataFrame.getColumnClass(column) == Long::class -> {
                    sb.append("long")
                }
                dataFrame.getColumnClass(column) == Int::class -> {
                    sb.append("int")
                }
                dataFrame.getColumnClass(column) == Boolean::class -> {
                    sb.append("boolean")
                }
                dataFrame.getColumnClass(column) == DateTime::class -> {
                    sb.append("timestamp")
                    if(timestamp == null) {
                        timestamp = dataFrame.getColumnName(column)
                    }
                }
                dataFrame.getColumnClass(column) == String::class -> {
        //                sb.append("string")
                    sb.append("symbol")
                }
                else -> {
        //                sb.append("string")
                    sb.append("symbol")
                }
            }
        }
        sb.append(")")
        if(timestamp != null) {
            sb.append("timestamp($timestamp) PARTITION BY MONTH")
        }

        compiler.compile(
            sb.toString(),
            ctx
        )
    }

    @Synchronized
    override fun <R, C, V> write(dataFrame: DataFrame<R, C, V>) {
        if(!initialized) {
            if(engine.tableNotExists(table)) {
                initialize(dataFrame)
            }

            initialized = true
        }

        var timestamp : C? = null
        for (column in dataFrame.columns()) {
            if(dataFrame.getColumnClass(column) == DateTime::class) {
                if(timestamp == null) {
                    timestamp = column
                }
            }
        }
        val tableToken = engine.getTableTokenIfExists(table)
        engine.getWriter(tableToken, "test").use { writer ->
            for (r in dataFrame.rows()) {
                val row = writer.newRow(if(timestamp != null) {(dataFrame.getValueAt(r, timestamp) as DateTime).unixMillisLong * 1000 } else {Os.currentTimeMicros()})

                for (column in dataFrame.columns()) {
                    val c = dataFrame.getColumnAddress(column)
                    val v = dataFrame.getValueAt(r, column)

                    when {
                        dataFrame.getColumnClass(column) == Double::class -> {
                            if(v is Double) {
                                row.putDouble(c, v)
                            } else if(v is Quantity<*>) {
                                row.putDouble(c, v.amount)
                            }
                        }
                        dataFrame.getColumnClass(column) == Float::class -> {
                            if(v is Float) {
                                row.putFloat(c, v)
                            } else if(v is Quantity<*>) {
                                row.putFloat(c, v.amount.toFloat())
                            }
                        }
                        dataFrame.getColumnClass(column) == Long::class -> {
                            if(v is Long) {
                                row.putLong(c, v)
                            }
                        }
                        dataFrame.getColumnClass(column) == Int::class -> {
                            if(v is Int) {
                                row.putInt(c, v)
                            }
                        }
                        dataFrame.getColumnClass(column) == Boolean::class -> {
                            if(v is Boolean) {
                                row.putBool(c, v)
                            }
                        }
                        dataFrame.getColumnClass(column) == DateTime::class -> {
                            // If this is the designated timestamp column, then the value is automatically populated by the call to newRow()
                            if (column != timestamp) {
                                if (v is DateTime) {
                                    row.putDate(c, v.unixMillisLong * 1000)
                                }
                            }
                        }
                        dataFrame.getColumnClass(column) == String::class -> {
                            if(v is String) {
                //                            row.putStr(c, v)
                                row.putSym(c, v)
                            } else {
                                if(v != null) {
                                    //                            row.putStr(c, v.toString())
                                    row.putSym(c, v.toString())
                                }
                            }
                        }
                        else -> {
                            if(v != null) {
                //                            row.putStr(c, v.toString())
                                row.putSym(c, v.toString())
                            }
                        }
                    }
                }
                row.append()
            }

            writer.commit()
        }
    }

    override fun close() {
    }
}