@file:OptIn(ExperimentalTime::class)

package org.molap.questdb

import com.macrofocus.common.json.*
import korlibs.time.DateFormat
import korlibs.time.DateTime
import korlibs.time.parseUtc
import kotlin.time.Instant
import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.geom.LineString
import org.locationtech.jts.io.WKTReader
import org.molap.dataframe.AbstractDataFrame
import org.molap.index.DefaultUniqueIndex
import org.molap.index.IntegerRangeUniqueIndex
import org.molap.index.UniqueIndex
import kotlin.reflect.KClass
import kotlin.time.ExperimentalTime

enum class DateTimeHandling {
    String,
    Long,
    DateTime,
    Instant
}

class QuestDBDataFrame(val root: JsonObject, val dateTimeHanding: DateTimeHandling = DateTimeHandling.String, vararg geomColumns: String): AbstractDataFrame<Int, String, Any?>() {
    private val labels: Array<String?>
    private val classes: Array<KClass<out Any>?>
    private val array: JsonArray?

    constructor(json: String, dateTimeHanding: DateTimeHandling = DateTimeHandling.String, vararg geomColumns: String) : this(JsonFactory.parse<JsonObject>(json), dateTimeHanding, *geomColumns) {
    }

    init {
        val columns = LinkedHashSet<String>()
        val types: HashMap<String, KClass<*>> = HashMap<String, KClass<*>>()

        val jsonColumns = root.getArray("columns")

        if(jsonColumns != null) {
            for (i in 0 until jsonColumns.length()) {
                val column = jsonColumns.getObject(i)
                val name = column?.getString("name")!!
                val type = column.getString("type")!!.lowercase()

                columns.add(name)
                types[name] = when {
                    type == "boolean" -> Boolean::class
                    type == "byte" -> Number::class
                    type == "short" -> Number::class
                    type == "char" -> String::class
                    type == "int" -> Int::class
                    type == "float" -> Float::class
                    type == "symbol" -> if (geomColumns.contains(name)) Geometry::class else String::class
                    type == "string" -> if (geomColumns.contains(name)) Geometry::class else String::class
                    type == "varchar" -> if (geomColumns.contains(name)) Geometry::class else String::class
                    type == "long" -> Long::class
                    type == "date" -> when (dateTimeHanding) {
                        DateTimeHandling.String -> String::class
                        DateTimeHandling.Long -> Long::class
                        DateTimeHandling.DateTime -> DateTime::class
                        DateTimeHandling.Instant -> Instant::class
                    }

                    type == "timestamp" -> when (dateTimeHanding) {
                        DateTimeHandling.String -> String::class
                        DateTimeHandling.Long -> Long::class
                        DateTimeHandling.DateTime -> DateTime::class
                        DateTimeHandling.Instant -> Instant::class
                    }

                    type == "double" -> Double::class
                    type == "binary" -> String::class
                    type == "long256" -> Number::class
                    type.startsWith("geohash") -> String::class
                    else -> {
                        println("Unknown type: $type for column $name. Assuming Any::class.")
                        Any::class
                    }
                }
            }
        }

        array = root.getArray("dataset")

        labels = arrayOfNulls(columns.size)
        classes = arrayOfNulls<KClass<out Any>>(columns.size)
        var index = 0
        for (column in columns) {
            labels[index] = column
            val type: KClass<*>? = types[column]
            classes[index] = if (type != null) type else Any::class
            index++
        }
    }

    override fun getRowClass(row: Int): KClass<*> {
        return Any::class
    }

    override fun getColumnClass(column: String): KClass<out Any> {
        return classes[columnIndex.getAddress(column)]!!
    }

    override fun getValueAt(row: Int, column: String): Any? {
        return getValueAt(getRowAddress(row), getColumnAddress(column))
    }

    override val rowIndex: UniqueIndex<Int> by lazy { IntegerRangeUniqueIndex(0, (array?.length() ?: 0) - 1) }
    override val columnIndex: UniqueIndex<String> by lazy {
        val names = ArrayList<String>(labels.size)
        for (c in labels.indices) {
            names.add(labels[c]!!)
        }
        DefaultUniqueIndex<String>(names)
    }

    fun getValueAt(rowIndex: Int, columnIndex: Int): Any? {
        return try {
            val value: JsonValue? = array?.getArray(rowIndex)?.get(columnIndex)
            if(value != null) {
                if (value.type === JsonType.NULL) {
                    null
                } else if (value.type == JsonType.ARRAY) {
                    val array: JsonArray = value.asJsonArray()!!
                    val a = arrayOfNulls<Any>(array.length())
                    for (i in 0 until array.length()) {
                        a[i] = array.get(i)
                    }
                    a
                } else if (value.type == JsonType.BOOLEAN) {
                    value.asBoolean()
                } else if (value.type == JsonType.NUMBER) {
                    when {
                        classes[columnIndex] == Int::class -> value.asNumber().toInt()
                        classes[columnIndex] == Float::class -> value.asNumber().toFloat()
                        classes[columnIndex] == Long::class -> value.asNumber().toLong()
                        else -> value.asNumber()
                    }
                } else if (value.type == JsonType.STRING) {
                    if(classes[columnIndex] == DateTime::class) {
                        value.asString()?.let {
                            dateFormat.parseUtc(it)
                        }
                    } else if(classes[columnIndex] == Long::class) {
                        value.asString()?.let {
                            dateFormat.parseUtc(it).unixMillisLong
                        }
                    } else if(classes[columnIndex] == Instant::class) {
                        value.asString()?.let {
                            Instant.fromEpochMilliseconds(dateFormat.parseUtc(it).unixMillisLong)
                        }
                    } else if(classes[columnIndex] == Geometry::class) {
                        value.asString()?.let {
                            val geometry = WKTReader().read(it)

                            // Some lines may have 0 length
                            if(geometry is LineString && geometry.length == 0.0) {
                                // Tableau doesn't support sending back a point
                                // Unable to complete action
                                // class ComputeModelTask::Execute() caught std::exception: The spatial operation resulted in a MixedGeometry or MixedGeography, which Tableau does not support yet.
                                // Error Code: 58D44822
//                                geometry.startPoint
                                null
                            } else {
                                geometry
                            }
                        }
                    } else {
                        value.asString()
                    }
                } else {
                    value
                }
            } else {
                return null
            }
        } catch (e: JsonException) {
            e.printStackTrace()
            null
        }
    }

    companion object {
        val dateFormat: DateFormat by lazy { DateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSz") }
    }
}