@file:OptIn(ExperimentalTime::class)

package org.molap.questdb

import com.macrofocus.common.units.Quantity
import io.questdb.cairo.CairoEngine
import io.questdb.std.BinarySequence
import io.questdb.std.Os
import io.questdb.std.str.Utf8String
import korlibs.time.DateTime
import kotlin.time.Instant
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toInstant
import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.geom.LineString
import org.locationtech.jts.geom.Point
import org.locationtech.jts.geom.Polygon
import org.locationtech.jts.io.WKTWriter
import org.molap.dataframe.DataFrame
import org.molap.datetime.DateTimeTz
import org.molap.exporter.DataFrameExport
import java.time.ZoneOffset
import kotlin.time.ExperimentalTime

class QuestDBDataFrameExport(
    private val engine: CairoEngine,
    val table: String,
    private val lock: String = "$table-lock",
    initialize: Boolean = true
) : DataFrameExport {
    private var initialized = !initialize
//    val ctx = SqlExecutionContextImpl(engine, 1)
//    val compiler = SqlCompilerImpl(engine)
//    init {
//        ctx.with(AllowAllSecurityContext.INSTANCE, null, null)
//    }

    private fun <R, C, V> initialize(dataFrame: DataFrame<R, C, V>) {
        val sb = StringBuilder("create table \"$table\" (")
        var first = true
        var timestamp: String? = null
        for (column in dataFrame.columns()) {
            if (!first) {
                sb.append(", ")
            } else {
                first = false
            }
            sb.append('"' + dataFrame.getColumnName(column)!! + '"')
            sb.append(" ")
            when {
                dataFrame.getColumnClass(column) == Double::class -> {
                    sb.append("double")
                }

                dataFrame.getColumnClass(column) == Quantity::class -> {
                    sb.append("double")
                }

                dataFrame.getColumnClass(column) == Float::class -> {
                    sb.append("float")
                }

                dataFrame.getColumnClass(column) == Long::class -> {
                    sb.append("long")
                }

                dataFrame.getColumnClass(column) == Int::class -> {
                    sb.append("int")
                }

                dataFrame.getColumnClass(column) == Boolean::class -> {
                    sb.append("boolean")
                }

                dataFrame.getColumnClass(column) == DateTime::class -> {
                    sb.append("timestamp")
                    if (timestamp == null) {
                        timestamp = dataFrame.getColumnName(column)
                    }
                }

                dataFrame.getColumnClass(column) == DateTimeTz::class -> {
                    sb.append("timestamp")
                    if (timestamp == null) {
                        timestamp = dataFrame.getColumnName(column)
                    }
                }

                dataFrame.getColumnClass(column) == kotlin.time.Instant::class -> {
                    sb.append("timestamp")
                    if (timestamp == null) {
                        timestamp = dataFrame.getColumnName(column)
                    }
                }

                dataFrame.getColumnClass(column) == kotlinx.datetime.LocalDateTime::class -> {
                    sb.append("timestamp")
                    if (timestamp == null) {
                        timestamp = dataFrame.getColumnName(column)
                    }
                }

                dataFrame.getColumnClass(column) == java.time.LocalDateTime::class -> {
                    sb.append("timestamp")
                    if (timestamp == null) {
                        timestamp = dataFrame.getColumnName(column)
                    }
                }

                dataFrame.getColumnClass(column) == java.sql.Timestamp::class -> {
                    sb.append("timestamp")
                    if (timestamp == null) {
                        timestamp = dataFrame.getColumnName(column)
                    }
                }

                dataFrame.getColumnClass(column) == ByteArray::class -> {
                    sb.append("binary")
                }

                dataFrame.getColumnClass(column) == Geometry::class -> {
                    sb.append("varchar")
                }

                dataFrame.getColumnClass(column) == Point::class -> {
                    sb.append("varchar")
                }

                dataFrame.getColumnClass(column) == LineString::class -> {
                    sb.append("varchar")
                }

                dataFrame.getColumnClass(column) == Polygon::class -> {
                    sb.append("varchar")
                }

                dataFrame.getColumnClass(column) == String::class -> {
                    //                sb.append("string")
                    sb.append("symbol INDEX")
                }

                else -> {
                    //                sb.append("string")
                    println("Unknown column $column with type ${dataFrame.getColumnClass(column)}")
                    sb.append("varchar")
                }
            }
        }
        sb.append(")")
        if (timestamp != null) {
            sb.append("timestamp($timestamp) PARTITION BY WEEK")
        }
        sb.append(";")

        engine.execute(sb.toString())
    }

    fun truncate() {
        if (!engine.tableNotExists(table)) {
           engine.execute("TRUNCATE TABLE $table")
        }
    }

    @Synchronized
    override fun <R, C, V> write(dataFrame: DataFrame<R, C, V>) {
        if (!initialized) {
            if (engine.tableNotExists(table)) {
                initialize(dataFrame)
            }

            initialized = true
        }

        var timestamp: C? = null
        for (column in dataFrame.columns()) {
            if (dataFrame.getColumnClass(column) == DateTime::class) {
                if (timestamp == null) {
                    timestamp = column
                }
            } else if(dataFrame.getColumnClass(column) == DateTime::class) {
                if (timestamp == null) {
                    timestamp = column
                }
            } else if(dataFrame.getColumnClass(column) == DateTimeTz::class) {
                if (timestamp == null) {
                    timestamp = column
                }
            } else if(dataFrame.getColumnClass(column) == kotlin.time.Instant::class) {
                if (timestamp == null) {
                    timestamp = column
                }
            } else if(dataFrame.getColumnClass(column) == kotlinx.datetime.LocalDateTime::class) {
                if (timestamp == null) {
                    timestamp = column
                }
            } else if(dataFrame.getColumnClass(column) == java.time.LocalDateTime::class) {
                if (timestamp == null) {
                    timestamp = column
                }
            } else if(dataFrame.getColumnClass(column) == java.sql.Timestamp::class) {
                if (timestamp == null) {
                    timestamp = column
                }
            }

        }
        val tableToken = engine.getTableTokenIfExists(table)
        engine.getWriter(tableToken, lock).use { writer ->
            for (r in dataFrame.rows()) {
                val row = writer.newRow(
                    if (timestamp != null) {
                        val value = dataFrame.getValueAt(r, timestamp)
                        if (value is DateTime) {
                            (value as DateTime).unixMillisLong * 1000
                        } else if (value is DateTimeTz) {
                            (value as DateTimeTz).datetime.toEpochMilliseconds() * 1000
                        } else if (value is kotlinx.datetime.LocalDateTime) {
                            (value as kotlinx.datetime.LocalDateTime).toInstant(TimeZone.UTC).toEpochMilliseconds() * 1000
                        } else if (value is kotlin.time.Instant) {
                            (value as kotlin.time.Instant).toEpochMilliseconds() * 1000
                        } else if (value is java.time.LocalDateTime) {
                            (value as java.time.LocalDateTime).toInstant(ZoneOffset.UTC).toEpochMilli() * 1000
                        } else if (value is java.sql.Timestamp) {
                            (value as java.sql.Timestamp).time * 1000
                        } else {
                            throw UnsupportedOperationException("Unsupported value type: ${value!!::class.qualifiedName}")
                        }
                    } else {
                        Os.currentTimeMicros()
                    })


                    for (column in dataFrame.columns()) {
                        val c = dataFrame.getColumnAddress(column)
                        val v = dataFrame.getValueAt(r, column)

                        when {
                            dataFrame.getColumnClass(column) == Double::class -> {
                                if (v is Double) {
                                    row.putDouble(c, v)
                                } else if (v is Quantity<*>) {
                                    row.putDouble(c, v.amount)
                                }
                            }

                            dataFrame.getColumnClass(column) == Quantity::class -> {
                                if (v is Double) {
                                    row.putDouble(c, v)
                                } else if (v is Quantity<*>) {
                                    row.putDouble(c, v.amount)
                                }
                            }

                            dataFrame.getColumnClass(column) == Float::class -> {
                                if (v is Float) {
                                    row.putFloat(c, v)
                                } else if (v is Quantity<*>) {
                                    row.putFloat(c, v.amount.toFloat())
                                }
                            }

                            dataFrame.getColumnClass(column) == Long::class -> {
                                if (v is Long) {
                                    row.putLong(c, v)
                                }
                            }

                            dataFrame.getColumnClass(column) == Int::class -> {
                                if (v is Int) {
                                    row.putInt(c, v)
                                }
                            }

                            dataFrame.getColumnClass(column) == Boolean::class -> {
                                if (v is Boolean) {
                                    row.putBool(c, v)
                                }
                            }

                            dataFrame.getColumnClass(column) == DateTimeTz::class -> {
                                // If this is the designated timestamp column, then the value is automatically populated by the call to newRow()
                                if (column != timestamp) {
                                    if (v is DateTimeTz) {
                                        row.putDate(c, v.datetime.toEpochMilliseconds() * 1000)
                                    }
                                }
                            }

                            dataFrame.getColumnClass(column) == Instant::class -> {
                                // If this is the designated timestamp column, then the value is automatically populated by the call to newRow()
                                if (column != timestamp) {
                                    if (v is Instant) {
                                        row.putDate(c, v.toEpochMilliseconds() * 1000)
                                    }
                                }
                            }

                            dataFrame.getColumnClass(column) == java.time.LocalDateTime::class -> {
                                // If this is the designated timestamp column, then the value is automatically populated by the call to newRow()
                                if (column != timestamp) {
                                    if (v is java.time.LocalDateTime) {
                                        row.putDate(c, v.toInstant(ZoneOffset.UTC).toEpochMilli() * 1000)
                                    }
                                }
                            }

                            dataFrame.getColumnClass(column) == DateTime::class -> {
                                // If this is the designated timestamp column, then the value is automatically populated by the call to newRow()
                                if (column != timestamp) {
                                    if (v is DateTime) {
                                        row.putDate(c, v.unixMillisLong * 1000)
                                    }
                                }
                            }

                            dataFrame.getColumnClass(column) == java.sql.Timestamp::class -> {
                                // If this is the designated timestamp column, then the value is automatically populated by the call to newRow()
                                if (column != timestamp) {
                                    if (v is java.sql.Timestamp) {
                                        row.putDate(c, v.time * 1000)
                                    }
                                }
                            }

                            dataFrame.getColumnClass(column) == ByteArray::class -> {
                                if (v is ByteArray) {
                                    val binarySequence = object : BinarySequence {
                                        override fun length(): Long = v.size.toLong()
                                        override fun byteAt(index: Long): Byte = v[index.toInt()]
                                    }
                                    row.putBin(c, binarySequence)
                                }
                            }

                            dataFrame.getColumnClass(column) == Geometry::class -> {
                                if (v is Geometry) {
                                    row.putVarchar(c, Utf8String(WKTWriter().write(v)))
                                }
                            }

                            dataFrame.getColumnClass(column) == Point::class -> {
                                if (v is Point) {
                                    row.putVarchar(c, Utf8String(WKTWriter().write(v)))
                                }
                            }

                            dataFrame.getColumnClass(column) == LineString::class -> {
                                if (v is LineString) {
                                    row.putVarchar(c, Utf8String(WKTWriter().write(v)))
                                }
                            }

                            dataFrame.getColumnClass(column) == Polygon::class -> {
                                if (v is Polygon) {
                                    row.putVarchar(c, Utf8String(WKTWriter().write(v)))
                                }
                            }

                            dataFrame.getColumnClass(column) == String::class -> {
                                if (v is String) {
                                    //                            row.putStr(c, v)
                                    try {
                                        row.putSym(c, v)
                                    } catch (e: NullPointerException) {
                                        println("Cannot write value $v to column $column into table $table for row $row because of ${e.message}")
                                    } catch (e: UnsupportedOperationException) {
                                        println("Cannot write value $v to column $column into table $table for row $row because of ${e.message}")
                                    }
                                } else {
                                    if (v != null) {
                                        //                            row.putStr(c, v.toString())
                                        row.putSym(c, v.toString())
                                    }
                                }
                            }

                            else -> {
                                if (v != null) {
                                    println("Unknown type ${v!!::class}  for value $v in column $column of table $table... trying anyway")
                                    //                            row.putStr(c, v.toString())
                                    row.putSym(c, v.toString())
                                }
                            }
                        }
                    }
                row.append()
            }

            writer.commit()
        }
    }

    override fun close() {
    }
}